<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Bank;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;

class BankController extends Controller
{
    /**
     * Display a listing of banks.
     */
    public function index()
    {
        $banks = Bank::ordered()->get(['*']);

        return Inertia::render('Admin/Banks/Index', [
            'banks' => $banks,
        ]);
    }

    /**
     * Show the form for creating a new bank.
     */
    public function create()
    {
        return Inertia::render('Admin/Banks/Create');
    }

    /**
     * Store a newly created bank.
     */
    public function store(Request $request)
    {
        $bankType = $request->input('bank_type', 'bank');

        // Validación base
        $rules = [
            'bank_name' => 'required|string|max:255',
            'bank_type' => 'required|in:bank,western_union',
            'logo' => 'nullable|image|mimes:png,jpg,jpeg,svg,webp|max:1024',
            'is_active' => 'boolean',
            'sort_order' => 'integer|min:0',
        ];

        // Validación condicional según tipo
        if ($bankType === 'western_union') {
            $rules['account_holder'] = 'required|string|max:255';
            $rules['phone_number'] = 'required|string|max:20';
        } else {
            $rules['account_number'] = 'required|string|max:50';
            $rules['account_type'] = 'required|in:savings,checking';
            $rules['account_holder'] = 'required|string|max:255';
            $rules['rnc'] = 'nullable|string|max:20';
            $rules['currency'] = 'required|string|size:3';
            $rules['instructions'] = 'nullable|string|max:1000';
        }

        $validated = $request->validate($rules);

        // Handle logo upload
        if ($request->hasFile('logo')) {
            $validated['logo'] = $request->file('logo')->store('banks', 'public');
        }

        $validated['is_active'] = $validated['is_active'] ?? true;
        $validated['sort_order'] = $validated['sort_order'] ?? Bank::max('sort_order') + 1;

        Bank::create($validated);

        $message = $bankType === 'western_union'
            ? 'Western Union agregado exitosamente.'
            : 'Cuenta bancaria creada exitosamente.';

        return redirect()->route('admin.banks.index')
            ->with('success', $message);
    }

    /**
     * Show the form for editing a bank.
     */
    public function edit(Bank $bank)
    {
        return Inertia::render('Admin/Banks/Edit', [
            'bank' => $bank,
        ]);
    }

    /**
     * Update the specified bank.
     */
    public function update(Request $request, Bank $bank)
    {
        $bankType = $request->input('bank_type', $bank->bank_type ?? 'bank');

        // Validación base
        $rules = [
            'bank_name' => 'required|string|max:255',
            'bank_type' => 'required|in:bank,western_union',
            'logo' => 'nullable|image|mimes:png,jpg,jpeg,svg,webp|max:1024',
            'is_active' => 'boolean',
            'sort_order' => 'integer|min:0',
        ];

        // Validación condicional según tipo
        if ($bankType === 'western_union') {
            $rules['account_holder'] = 'required|string|max:255';
            $rules['phone_number'] = 'required|string|max:20';
        } else {
            $rules['account_number'] = 'required|string|max:50';
            $rules['account_type'] = 'required|in:savings,checking';
            $rules['account_holder'] = 'required|string|max:255';
            $rules['rnc'] = 'nullable|string|max:20';
            $rules['currency'] = 'required|string|size:3';
            $rules['instructions'] = 'nullable|string|max:1000';
        }

        $validated = $request->validate($rules);

        // Handle logo upload
        if ($request->hasFile('logo')) {
            // Delete old logo
            if ($bank->logo) {
                Storage::disk('public')->delete($bank->logo);
            }
            $validated['logo'] = $request->file('logo')->store('banks', 'public');
        }

        // Handle logo removal
        if ($request->boolean('remove_logo') && $bank->logo) {
            Storage::disk('public')->delete($bank->logo);
            $validated['logo'] = null;
        }

        $bank->update($validated);

        return redirect()->route('admin.banks.index')
            ->with('success', 'Cuenta bancaria actualizada exitosamente.');
    }

    /**
     * Toggle bank active status.
     */
    public function toggleStatus(Bank $bank)
    {
        $bank->is_active = !$bank->is_active;
        $bank->save();

        $status = $bank->is_active ? 'activada' : 'desactivada';
        return back()->with('success', "Cuenta bancaria {$status} exitosamente.");
    }

    /**
     * Remove the specified bank.
     */
    public function destroy(Bank $bank)
    {
        // Delete logo if exists
        if ($bank->logo) {
            Storage::disk('public')->delete($bank->logo);
        }

        $bank->delete();

        return redirect()->route('admin.banks.index')
            ->with('success', 'Cuenta bancaria eliminada exitosamente.');
    }

    /**
     * Update banks order.
     */
    public function updateOrder(Request $request)
    {
        $validated = $request->validate([
            'banks' => 'required|array',
            'banks.*.id' => 'required|exists:banks,id',
            'banks.*.sort_order' => 'required|integer|min:0',
        ]);

        foreach ($validated['banks'] as $bankData) {
            Bank::where('id', '=', $bankData['id'])->update(['sort_order' => $bankData['sort_order']]);
        }

        return back()->with('success', 'Orden actualizado exitosamente.');
    }
}
