<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Invoice;
use App\Models\Subscription;
use App\Models\User;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Inertia\Response;

class DashboardController extends Controller
{
    /**
     * Display the admin dashboard with global reports.
     */
    public function index(Request $request): Response
    {
        $totalCoaches = User::coaches()->count();
        $pendingCoaches = User::coaches()->pending()->count();
        $activeSubscriptions = Subscription::active()->count();
        
        // Conversion Rate (Trial -> Paid)
        // Coaches who have at least one paid subscription
        $paidCoachesCount = User::coaches()
            ->whereHas('subscriptions', function ($query) {
                $query->whereHas('plan', function ($q) {
                    $q->where('slug', '!=', 'trial');
                })->where('status', 'active');
            })->count();

        $conversionRate = $totalCoaches > 0 ? round(($paidCoachesCount / $totalCoaches) * 100, 2) : 0;

        // Cumulative Billing
        $totalBilling = Invoice::where('status', 'paid')->sum('total');
        
        // Monthly Billing (Current Month)
        $monthlyBilling = Invoice::where('status', 'paid')
            ->whereMonth('created_at', now()->month)
            ->whereYear('created_at', now()->year)
            ->sum('total');

        return Inertia::render('Admin/Dashboard', [
            'stats' => [
                'total_coaches' => $totalCoaches,
                'pending_coaches' => $pendingCoaches,
                'active_subscriptions' => $activeSubscriptions,
                'conversion_rate' => $conversionRate,
                'total_billing' => number_format($totalBilling, 2),
                'monthly_billing' => number_format($monthlyBilling, 2),
            ],
            'recent_subscriptions' => Subscription::with(['user', 'plan'])
                ->latest()
                ->take(5)
                ->get(),
        ]);
    }
}
