<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Traits\TogglesActiveStatus;
use App\Models\Plan;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Inertia\Response;

class PlanController extends Controller
{
    use TogglesActiveStatus;
    /**
     * Display a listing of plans.
     */
    public function index(): Response
    {
        $plans = Plan::withCount(['subscriptions' => function ($query) {
            $query->where('status', 'active');
        }])->get();

        return Inertia::render('Admin/Plans/Index', [
            'plans' => $plans,
        ]);
    }

    /**
     * Show the form for creating a new plan.
     */
    public function create(): Response
    {
        return Inertia::render('Admin/Plans/Create');
    }

    /**
     * Store a newly created plan.
     */
    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:plans',
            'price' => 'required|numeric|min:0',
            'duration_days' => 'required|integer|min:1',
            'description' => 'nullable|string|max:1000',
            'is_active' => 'boolean',
            'features' => 'required|array',
            'features.max_clients' => 'required|integer|min:-1',
            'features.max_diets' => 'required|integer|min:-1',
            'features.max_routines' => 'required|integer|min:-1',
            'features.custom_foods' => 'boolean',
            'features.pdf_export' => 'boolean',
            'features.analytics' => 'required|string|in:none,basic,advanced',
            'features.support' => 'required|string|in:none,email,priority',
        ]);

        Plan::create($validated);

        return redirect()->route('admin.plans.index')
            ->with('success', 'Plan creado correctamente.');
    }

    /**
     * Show the form for editing the specified plan.
     */
    public function edit(Plan $plan): Response
    {
        return Inertia::render('Admin/Plans/Edit', [
            'plan' => $plan,
        ]);
    }

    /**
     * Update the specified plan.
     */
    public function update(Request $request, Plan $plan): RedirectResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:plans,slug,' . $plan->id,
            'price' => 'required|numeric|min:0',
            'duration_days' => 'required|integer|min:1',
            'description' => 'nullable|string|max:1000',
            'is_active' => 'boolean',
            'features' => 'required|array',
            'features.max_clients' => 'required|integer|min:-1',
            'features.max_diets' => 'required|integer|min:-1',
            'features.max_routines' => 'required|integer|min:-1',
            'features.custom_foods' => 'boolean',
            'features.pdf_export' => 'boolean',
            'features.analytics' => 'required|string|in:none,basic,advanced',
            'features.support' => 'required|string|in:none,email,priority',
        ]);

        $plan->update($validated);

        return redirect()->route('admin.plans.index')
            ->with('success', 'Plan actualizado correctamente.');
    }

    /**
     * Toggle plan active status.
     */
    public function toggleActiveStatus(Plan $plan): RedirectResponse
    {
        $plan->update(['is_active' => !$plan->is_active]);

        $status = $plan->is_active ? 'activado' : 'desactivado';
        
        return back()->with('success', "El plan ha sido {$status}.");
    }

    /**
     * Toggle plan featured status.
     */
    public function toggleFeatured(Plan $plan): RedirectResponse
    {
        $plan->update(['is_featured' => !$plan->is_featured]);

        $status = $plan->is_featured ? 'marcado como popular' : 'desmarcado como popular';
        
        return back()->with('success', "El plan ha sido {$status}.");
    }

    /**
     * Remove the specified plan.
     */
    public function destroy(Plan $plan): RedirectResponse
    {
        // Prevent deletion if plan has active subscriptions
        if ($plan->subscriptions()->where('status', 'active')->exists()) {
            return back()->with('error', 'No se puede eliminar un plan con suscripciones activas.');
        }

        $plan->delete();

        return redirect()->route('admin.plans.index')
            ->with('success', 'Plan eliminado correctamente.');
    }
}
