<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;
use Inertia\Response;

class SettingsController extends Controller
{
    /**
     * Display the settings page.
     */
    public function index(): Response
    {
        // Get email settings from .env file (not database)
        $emailSettings = [
            'mail_host' => env('MAIL_HOST', ''),
            'mail_port' => env('MAIL_PORT', 587),
            'mail_username' => env('MAIL_USERNAME', ''),
            'mail_password' => env('MAIL_PASSWORD', ''),
            'mail_from_address' => env('MAIL_FROM_ADDRESS', ''),
            'mail_from_name' => env('MAIL_FROM_NAME', config('app.name')),
            'mail_encryption' => env('MAIL_ENCRYPTION', 'tls'),
            'mail_password_set' => !empty(env('MAIL_PASSWORD')),
        ];

        // Get payment settings and mask sensitive fields
        $paymentSettings = Setting::getGroup('payment');
        if (isset($paymentSettings['paypal_secret']) && !empty($paymentSettings['paypal_secret'])) {
            $paymentSettings['paypal_secret_set'] = true;
            unset($paymentSettings['paypal_secret']);
        }
        if (isset($paymentSettings['stripe_secret']) && !empty($paymentSettings['stripe_secret'])) {
            $paymentSettings['stripe_secret_set'] = true;
            unset($paymentSettings['stripe_secret']);
        }

        // Get general settings and ensure system_email has a fallback from .env
        $generalSettings = Setting::getGroup('general');
        if (empty($generalSettings['system_email'])) {
            $generalSettings['system_email'] = env('MAIL_FROM_ADDRESS', '');
        }

        return Inertia::render('Admin/Settings/Index', [
            'settings' => [
                'general' => $generalSettings,
                'seo' => Setting::getGroup('seo'),
                'email' => $emailSettings,
                'pdf' => Setting::getGroup('pdf'),
                'payment' => $paymentSettings,
                'cron' => Setting::getGroup('cron'),
                'security' => Setting::getGroup('security'),
                'notifications' => Setting::getGroup('notifications'),
                'pwa' => Setting::getGroup('pwa'),
                'features' => Setting::getGroup('features'),
            ],
        ]);
    }

    /**
     * Update notification settings (toggles).
     */
    public function updateNotifications(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'notify_new_user' => 'boolean',
            'notify_invoice_paid' => 'boolean',
            'notify_subscription_expired' => 'boolean',
            'notify_new_order' => 'boolean',
        ]);

        foreach ($validated as $key => $value) {
            Setting::set($key, $value, 'notifications');
        }

        return back()->with('success', 'Configuración de notificaciones actualizada.');
    }

    /**
     * Update general settings.
     */
    public function updateGeneral(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'site_name' => 'required|string|max:255',
            'site_description' => 'nullable|string|max:500',
            'system_email' => 'required|email|max:255',
            'timezone' => 'required|string',
            'currency' => 'required|string|size:3',
            'language' => 'required|string|size:2',
            'contact_email' => 'nullable|email|max:255',
            'contact_phone' => 'nullable|string|max:50',
            'business_hours' => 'nullable|string|max:255',
        ], [
            'system_email.required' => 'El correo del sistema es obligatorio.',
            'system_email.email' => 'El correo del sistema debe ser un email válido.',
            'contact_email.email' => 'El correo de contacto debe ser un email válido.',
            'contact_phone.max' => 'El teléfono no puede exceder 50 caracteres.',
        ]);

        foreach ($validated as $key => $value) {
            Setting::set($key, $value, 'general');
        }

        // Also update .env MAIL_FROM_ADDRESS if system_email is set
        if (!empty($validated['system_email'])) {
            $this->updateEnvValue('MAIL_FROM_ADDRESS', $validated['system_email']);
        }

        return back()->with('success', 'Configuración general actualizada.');
    }

    /**
     * Update a value in .env file
     */
    protected function updateEnvValue(string $key, string $value): void
    {
        $envPath = base_path('.env');
        $envContent = file_get_contents($envPath);

        // Handle values that need quoting
        $envValue = $value;
        if (str_contains($value, ' ') || str_contains($value, '#') || str_contains($value, '"')) {
            $envValue = '"' . addslashes($value) . '"';
        }

        if (preg_match("/^{$key}=.*/m", $envContent)) {
            $envContent = preg_replace("/^{$key}=.*/m", "{$key}={$envValue}", $envContent);
        } else {
            $envContent .= "\n{$key}={$envValue}";
        }

        file_put_contents($envPath, $envContent);
        Artisan::call('config:clear');
    }

    /**
     * Update cron job settings.
     */
    public function updateCron(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'cron_billing_enabled' => 'boolean',
            'cron_reminders_enabled' => 'boolean',
        ]);

        foreach ($validated as $key => $value) {
            Setting::set($key, $value, 'cron');
        }

        return back()->with('success', 'Configuración de tareas programadas actualizada.');
    }

    /**
     * Update security settings (reCAPTCHA).
     */
    public function updateSecurity(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'recaptcha_enabled' => 'boolean',
            'recaptcha_key' => 'nullable|string|max:255',
            'recaptcha_secret' => 'nullable|string|max:255',
        ]);

        foreach ($validated as $key => $value) {
            Setting::set($key, $value, 'security');
        }

        return back()->with('success', 'Configuración de seguridad actualizada.');
    }

    /**
     * Update SEO settings.
     */
    public function updateSeo(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'seo_title' => 'nullable|string|max:60',
            'seo_description' => 'nullable|string|max:160',
            'seo_keywords' => 'nullable|string|max:255',
            'google_analytics_id' => 'nullable|string|max:50',
            'google_tag_manager_id' => 'nullable|string|max:50',
            'facebook_pixel_id' => 'nullable|string|max:50',
            'google_site_verification' => 'nullable|string|max:100',
            'bing_site_verification' => 'nullable|string|max:100',
            'robots_meta' => 'nullable|string|max:50',
            'custom_header_scripts' => 'nullable|string',
            'custom_footer_scripts' => 'nullable|string',
        ]);

        foreach ($validated as $key => $value) {
            Setting::set($key, $value, 'seo');
        }

        return back()->with('success', 'Configuración SEO y Scripts actualizada.');
    }

    /**
     * Upload Open Graph Image.
     */
    public function uploadOgImage(Request $request): RedirectResponse
    {
        $request->validate([
            'og_image' => 'required|image|mimes:png,jpg,jpeg|max:10240',
        ]);

        $oldImage = Setting::get('og_image');
        if ($oldImage && Storage::disk('public')->exists($oldImage)) {
            Storage::disk('public')->delete($oldImage);
        }

        $path = $request->file('og_image')->store('settings', 'public');
        Setting::set('og_image', $path, 'seo');

        return back()->with('success', 'Imagen Open Graph actualizada.');
    }

    /**
     * Update email/SMTP settings directly in .env file.
     */
    public function updateEmail(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'mail_from_address' => 'required|email',
            'mail_from_name' => 'required|string|max:255',
            'mail_host' => 'nullable|string|max:255',
            'mail_port' => 'nullable|integer|min:1|max:65535',
            'mail_username' => 'nullable|string|max:255',
            'mail_password' => 'nullable|string|max:255',
            'mail_encryption' => 'nullable|string|in:tls,ssl,null',
        ]);

        // Auto-determine encryption based on port if not specified
        if (!empty($validated['mail_port']) && empty($validated['mail_encryption'])) {
            $port = (int) $validated['mail_port'];
            $validated['mail_encryption'] = match($port) {
                465 => 'ssl',
                587, 25, 2525 => 'tls',
                default => 'tls',
            };
        }

        // Map form fields to .env variable names
        $envMapping = [
            'mail_host' => 'MAIL_HOST',
            'mail_port' => 'MAIL_PORT',
            'mail_username' => 'MAIL_USERNAME',
            'mail_password' => 'MAIL_PASSWORD',
            'mail_from_address' => 'MAIL_FROM_ADDRESS',
            'mail_from_name' => 'MAIL_FROM_NAME',
            'mail_encryption' => 'MAIL_ENCRYPTION',
        ];

        $envPath = base_path('.env');
        $envContent = file_get_contents($envPath);

        foreach ($validated as $key => $value) {
            // Skip empty password to preserve existing
            if ($key === 'mail_password' && empty($value)) {
                continue;
            }

            $envKey = $envMapping[$key] ?? null;
            if (!$envKey) {
                continue;
            }

            // Handle values that need quoting (spaces, special chars)
            $envValue = $value;
            if (is_string($value) && (str_contains($value, ' ') || str_contains($value, '#') || str_contains($value, '"'))) {
                $envValue = '"' . addslashes($value) . '"';
            }

            // Update or add the environment variable
            if (preg_match("/^{$envKey}=.*/m", $envContent)) {
                $envContent = preg_replace("/^{$envKey}=.*/m", "{$envKey}={$envValue}", $envContent);
            } else {
                $envContent .= "\n{$envKey}={$envValue}";
            }
        }

        // Ensure MAIL_MAILER is set to smtp
        if (!preg_match("/^MAIL_MAILER=smtp/m", $envContent)) {
            $envContent = preg_replace("/^MAIL_MAILER=.*/m", "MAIL_MAILER=smtp", $envContent);
        }

        file_put_contents($envPath, $envContent);

        // Clear config cache to apply changes
        Artisan::call('config:clear');

        return back()->with('success', 'Configuración SMTP guardada en .env correctamente.');
    }

    /**
     * Update PDF settings.
     */
    public function updatePdf(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'pdf_header' => 'nullable|string|max:255',
            'pdf_footer' => 'nullable|string|max:255',
            'pdf_note' => 'nullable|string|max:500',
        ]);

        foreach ($validated as $key => $value) {
            Setting::set($key, $value, 'pdf');
        }

        return back()->with('success', 'Configuración de PDF actualizada.');
    }

    /**
     * Update payment gateway settings.
     */
    public function updatePayment(Request $request): RedirectResponse
    {
        // Merge boolean defaults for unchecked checkboxes
        $request->merge([
            'paypal_enabled' => $request->boolean('paypal_enabled'),
            'paypal_sandbox' => $request->boolean('paypal_sandbox'),
            'paypal_fee_enabled' => $request->boolean('paypal_fee_enabled'),
            'stripe_enabled' => $request->boolean('stripe_enabled'),
            'stripe_test_mode' => $request->boolean('stripe_test_mode'),
            'stripe_fee_enabled' => $request->boolean('stripe_fee_enabled'),
            'bank_transfer_enabled' => $request->boolean('bank_transfer_enabled'),
        ]);

        $validated = $request->validate([
            'paypal_enabled' => 'boolean',
            'paypal_client_id' => 'nullable|string|max:255',
            'paypal_secret' => 'nullable|string', // Removed max:255
            'paypal_sandbox' => 'boolean',
            'paypal_fee_enabled' => 'boolean',
            'stripe_enabled' => 'boolean',
            'stripe_key' => 'nullable|string|max:255',
            'stripe_secret' => 'nullable|string', // Removed max:255
            'stripe_test_mode' => 'boolean',
            'stripe_fee_enabled' => 'boolean',
            'bank_transfer_enabled' => 'boolean',
            'bank_details' => 'nullable|string|max:1000',
        ]);

        foreach ($validated as $key => $value) {
            // Encrypt sensitive data - only if value is not empty
            if (in_array($key, ['paypal_secret', 'stripe_secret'])) {
                if (!empty($value)) {
                    $value = encrypt($value);
                } else {
                    // Keep existing encrypted value if new value is empty
                    continue;
                }
            }
            Setting::set($key, $value, 'payment');
        }

        return back()->with('success', 'Configuración de pagos actualizada.');
    }

    /**
     * Upload logo.
     */
    public function uploadLogo(Request $request): RedirectResponse
    {
        $request->validate([
            'logo' => 'required|image|mimes:png,jpg,jpeg,svg|max:10240',
        ]);

        // Delete old logo if exists
        $oldLogo = Setting::get('logo');
        if ($oldLogo && Storage::disk('public')->exists($oldLogo)) {
            Storage::disk('public')->delete($oldLogo);
        }

        $path = $request->file('logo')->store('settings', 'public');
        Setting::set('logo', $path, 'general');

        return back()->with('success', 'Logo actualizado.');
    }

    /**
     * Upload favicon.
     */
    public function uploadFavicon(Request $request): RedirectResponse
    {
        $request->validate([
            'favicon' => 'required|image|mimes:png,ico|max:10240',
        ]);

        $oldFavicon = Setting::get('favicon');
        if ($oldFavicon && Storage::disk('public')->exists($oldFavicon)) {
            Storage::disk('public')->delete($oldFavicon);
        }

        $path = $request->file('favicon')->store('settings', 'public');
        Setting::set('favicon', $path, 'general');

        return back()->with('success', 'Favicon actualizado.');
    }

    /**
     * Update PWA settings.
     */
    public function updatePwa(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'pwa_enabled' => 'boolean',
            'pwa_name' => 'nullable|string|max:255',
            'pwa_short_name' => 'nullable|string|max:50',
            'pwa_description' => 'nullable|string|max:500',
            'pwa_theme_color' => 'nullable|string|max:7',
            'pwa_background_color' => 'nullable|string|max:7',
            'pwa_display' => 'nullable|in:standalone,fullscreen,minimal-ui,browser',
        ]);

        foreach ($validated as $key => $value) {
            Setting::set($key, $value, 'pwa');
        }

        return back()->with('success', 'Configuración PWA actualizada.');
    }

    /**
     * Upload PWA icon (192x192 or 512x512, regular or maskable).
     * Saves directly to public/icons/png/ for Vite to detect.
     */
    public function uploadPwaIcon(Request $request): RedirectResponse
    {
        $request->validate([
            'pwa_icon' => 'required|image|mimes:png|max:10240',
            'size' => 'required|in:192,512',
            'type' => 'required|in:regular,maskable',
        ]);

        $size = $request->input('size');
        $type = $request->input('type');

        // Determine the filename based on type
        $filename = $type === 'maskable'
            ? "maskable-icon-{$size}x{$size}.png"
            : "icon-{$size}x{$size}.png";

        // Save directly to public/icons/png/ directory
        $publicPath = public_path('icons/png');
        if (!file_exists($publicPath)) {
            mkdir($publicPath, 0755, true);
        }

        $destinationPath = $publicPath . '/' . $filename;

        // Delete old icon if exists
        if (file_exists($destinationPath)) {
            unlink($destinationPath);
        }

        // Move uploaded file to public directory
        $request->file('pwa_icon')->move($publicPath, $filename);

        $iconLabel = $type === 'maskable' ? "maskable {$size}x{$size}" : "{$size}x{$size}";

        return back()->with('success', "Icono PWA {$iconLabel} actualizado. Se aplicará en el próximo build.");
    }

    /**
     * Upload Apple Touch Icon (180x180).
     */
    public function uploadAppleTouchIcon(Request $request): RedirectResponse
    {
        $request->validate([
            'apple_icon' => 'required|image|mimes:png|max:10240',
        ]);

        $oldIcon = Setting::get('pwa_apple_icon');
        if ($oldIcon && Storage::disk('public')->exists($oldIcon)) {
            Storage::disk('public')->delete($oldIcon);
        }

        $path = $request->file('apple_icon')->store('settings/pwa', 'public');
        Setting::set('pwa_apple_icon', $path, 'pwa');

        return back()->with('success', "Icono Apple Touch actualizado.");
    }

    /**
     * Upload PWA Splash Screen.
     */
    public function uploadPwaSplash(Request $request): RedirectResponse
    {
        $request->validate([
            'pwa_splash' => 'required|image|mimes:png,jpg,jpeg|max:10240',
        ]);

        $oldSplash = Setting::get('pwa_splash');
        if ($oldSplash && Storage::disk('public')->exists($oldSplash)) {
            Storage::disk('public')->delete($oldSplash);
        }

        $path = $request->file('pwa_splash')->store('settings/pwa', 'public');
        Setting::set('pwa_splash', $path, 'pwa');

        return back()->with('success', "Pantalla de inicio PWA actualizada.");
    }

    /**
     * Update feature toggles.
     */
    public function updateFeatures(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'feature_client_portal' => 'boolean',
            'feature_progress_photos' => 'boolean',
            'feature_wallet' => 'boolean',
            'feature_referrals' => 'boolean',
            'feature_pdf_export' => 'boolean',
            'feature_chat' => 'boolean',
            'feature_compliance_tracking' => 'boolean',
            'feature_analytics' => 'boolean',
            'feature_custom_foods' => 'boolean',
        ]);

        foreach ($validated as $key => $value) {
            Setting::set($key, $value, 'features');
        }

        return back()->with('success', 'Funcionalidades actualizadas.');
    }

    /**
     * Test SMTP connection and send a test email.
     */
    public function testEmail(Request $request)
    {
        try {
            $validated = $request->validate([
                'mail_host' => 'required|string',
                'mail_port' => 'required|integer',
                'mail_username' => 'nullable|string',
                'mail_password' => 'nullable|string',
                'mail_from_address' => 'required|email',
                'mail_from_name' => 'required|string',
                'test_email_address' => 'nullable|email',
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error de validación: ' . implode(', ', $e->validator->errors()->all()),
            ]);
        }

        try {
            // Get password: use provided one, or use one from .env
            $password = $validated['mail_password'];
            if (empty($password)) {
                // First try env(), then try reading directly from .env file
                $password = env('MAIL_PASSWORD');
                if (empty($password)) {
                    // Read directly from .env file as fallback
                    $envPath = base_path('.env');
                    if (file_exists($envPath)) {
                        $envContent = file_get_contents($envPath);
                        if (preg_match('/^MAIL_PASSWORD=["\'"]?(.+?)["\'"]?$/m', $envContent, $matches)) {
                            $password = trim($matches[1], '"\'');
                        }
                    }
                }
            }

            if (empty($password)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No se encontró la contraseña SMTP. Configura MAIL_PASSWORD en el archivo .env o ingresa la contraseña en el campo.',
                ]);
            }

            // Clear any cached config first
            Artisan::call('config:clear');

            // Configure temporary mailer with provided settings
            config([
                'mail.mailers.smtp.host' => $validated['mail_host'],
                'mail.mailers.smtp.port' => (int) $validated['mail_port'],
                'mail.mailers.smtp.username' => $validated['mail_username'],
                'mail.mailers.smtp.password' => $password,
                'mail.mailers.smtp.encryption' => $validated['mail_port'] == 465 ? 'ssl' : 'tls',
                'mail.from.address' => $validated['mail_from_address'],
                'mail.from.name' => $validated['mail_from_name'],
            ]);

            // Clear the mail manager to use new config
            app()->forgetInstance('mail.manager');

            // Send test email - use custom address if provided, otherwise use admin's email
            $targetEmail = !empty($validated['test_email_address'])
                ? $validated['test_email_address']
                : $request->user()->email;

            \Illuminate\Support\Facades\Mail::raw(
                "Este es un correo de prueba desde CoachPro.\n\n" .
                "Si recibes este mensaje, tu configuración SMTP está funcionando correctamente.\n\n" .
                "Fecha y hora: " . now()->format('d/m/Y H:i:s') . "\n" .
                "Host: {$validated['mail_host']}\n" .
                "Puerto: {$validated['mail_port']}",
                function ($message) use ($targetEmail, $validated) {
                    $message->to($targetEmail)
                            ->subject('Prueba de correo - CoachPro')
                            ->from($validated['mail_from_address'], $validated['mail_from_name']);
                }
            );

            return response()->json([
                'success' => true,
                'message' => "Correo de prueba enviado a {$targetEmail}. Revisa tu bandeja de entrada.",
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ]);
        }
    }
}
