<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\PushSubscription;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class PushSubscriptionController extends Controller
{
    /**
     * Get VAPID public key for push subscriptions.
     */
    public function vapidPublicKey(): JsonResponse
    {
        return response()->json([
            'publicKey' => config('services.push.vapid_public_key'),
        ]);
    }

    /**
     * Store a new push subscription.
     */
    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'endpoint' => 'required|string|max:500',
            'keys.p256dh' => 'required|string',
            'keys.auth' => 'required|string',
        ]);

        $userAgent = $request->userAgent();
        $deviceType = $this->detectDeviceType($userAgent);

        $subscription = PushSubscription::updateOrCreate(
            [
                'endpoint' => $validated['endpoint'],
            ],
            [
                'user_id' => $request->user()->id,
                'p256dh_key' => $validated['keys']['p256dh'],
                'auth_token' => $validated['keys']['auth'],
                'user_agent' => $userAgent,
                'device_type' => $deviceType,
                'is_active' => true,
                'last_used_at' => now(),
            ]
        );

        Log::info('Push subscription created/updated', [
            'user_id' => $request->user()->id,
            'device_type' => $deviceType,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Suscripción push registrada correctamente',
            'subscription_id' => $subscription->id,
        ]);
    }

    /**
     * Remove a push subscription.
     */
    public function destroy(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'endpoint' => 'required|string',
        ]);

        $deleted = PushSubscription::where('endpoint', $validated['endpoint'])
            ->where('user_id', $request->user()->id)
            ->delete();

        return response()->json([
            'success' => $deleted > 0,
            'message' => $deleted ? 'Suscripción eliminada' : 'Suscripción no encontrada',
        ]);
    }

    /**
     * Detect device type from user agent.
     */
    protected function detectDeviceType(?string $userAgent): string
    {
        if (!$userAgent) {
            return 'unknown';
        }

        $userAgent = strtolower($userAgent);

        if (str_contains($userAgent, 'mobile') || str_contains($userAgent, 'android')) {
            return str_contains($userAgent, 'tablet') ? 'tablet' : 'mobile';
        }

        if (str_contains($userAgent, 'ipad')) {
            return 'tablet';
        }

        if (str_contains($userAgent, 'iphone')) {
            return 'mobile';
        }

        return 'web';
    }

    /**
     * Test push notification for current user.
     */
    public function test(Request $request): JsonResponse
    {
        $user = $request->user();
        $subscriptions = PushSubscription::forUser($user->id);

        if ($subscriptions->isEmpty()) {
            return response()->json([
                'success' => false,
                'message' => 'No hay suscripciones push activas para este usuario',
            ], 404);
        }

        // Send test notification
        app(\App\Services\PushNotificationService::class)->sendToUser(
            $user,
            'Prueba de Notificación',
            '¡Las notificaciones push están funcionando correctamente!',
            [
                'type' => 'test',
                'url' => route('dashboard'),
            ]
        );

        return response()->json([
            'success' => true,
            'message' => 'Notificación de prueba enviada a ' . $subscriptions->count() . ' dispositivo(s)',
        ]);
    }
}
