<?php

namespace App\Http\Controllers\Coach;

use App\Http\Controllers\Controller;
use App\Http\Traits\ChatTrait;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Inertia\Response;

class ChatController extends Controller
{
    use ChatTrait;

    public function index(Request $request): Response
    {
        $coach = $request->user();

        // Get clients who have a portal user
        $clients = $coach->clients()->whereNotNull('client_user_id')->with('portalUser')->get();

        // Add Admin Support to the list
        $admin = User::where('role', 'admin')->first();

        $selectedUserId = $request->query('user');
        $messages = [];
        $selectedUser = null;

        if ($selectedUserId) {
            $selectedUser = User::find($selectedUserId);

            // Verify this user belongs to one of the coach's clients OR is the admin
            $isClient = $clients->contains(fn($c) => $c->client_user_id == $selectedUserId);
            $isAdmin = $admin && $admin->id == $selectedUserId;

            if (($isClient || $isAdmin) && $selectedUser) {
                $this->markMessagesAsRead($selectedUserId, $coach->id);
                $messages = $this->getConversation($coach->id, $selectedUserId);
            }
        }

        $contactList = $clients->map(function($client) {
            // Use avatar_url from portal user if available, otherwise from client
            $portalUser = $client->portalUser;
            $avatarUrl = $portalUser?->avatar_url ?? ($client->avatar ? asset('storage/' . $client->avatar) : null);

            return [
                'id' => $client->id,
                'name' => $client->name,
                'user_id' => $client->client_user_id,
                'avatar_url' => $avatarUrl,
                'has_unread' => $this->hasUnreadMessages($client->client_user_id, auth()->id()),
            ];
        })->toArray();

        // Prepend Admin if exists
        if ($admin) {
            array_unshift($contactList, [
                'id' => 'admin',
                'name' => 'Soporte CoachPro (Admin) 🛠️',
                'user_id' => $admin->id,
                'avatar_url' => $admin->avatar_url,
                'has_unread' => $this->hasUnreadMessages($admin->id, $coach->id),
            ]);
        }

        return Inertia::render('Coach/Chat/Index', [
            'clients' => $contactList,
            'messages' => $messages,
            'selectedUser' => $selectedUser ? [
                'id' => $selectedUser->id,
                'name' => $selectedUser->name,
                'avatar_url' => $selectedUser->avatar_url,
            ] : null,
        ]);
    }

    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'receiver_id' => 'required|exists:users,id',
            'content' => 'required|string|max:1000',
        ]);

        $receiver = User::find($request->receiver_id);
        $notifyAdminSupport = $receiver && $receiver->isAdmin();

        $this->createAndNotifyMessage(
            $request->user(),
            $request->receiver_id,
            $request->input('content'),
            $notifyAdminSupport
        );

        return back();
    }
}

