<?php

namespace App\Http\Controllers\Coach;

use App\Http\Controllers\Controller;
use App\Models\Food;
use App\Models\MealPlan;
use App\Models\MealItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

/**
 * Controller for managing meal plan templates (coach-level, not client-specific)
 */
class MealPlanTemplateController extends Controller
{
    /**
     * Display list of meal plan templates
     */
    public function index(Request $request)
    {
        $user = $request->user();

        $templates = MealPlan::whereNull('client_id')
            ->where(function ($query) use ($user) {
                $query->where('user_id', $user->id)
                      ->orWhere('is_official', true);
            })
            ->with(['mealItems.food'])
            ->when($request->search, function ($query, $search) {
                $query->where('name', 'like', "%{$search}%");
            })
            ->orderBy('is_official', 'desc') // Official first
            ->orderBy('created_at', 'desc')
            ->paginate(12)
            ->withQueryString();

        return Inertia::render('Coach/MealPlans/Templates/Index', [
            'templates' => $templates,
            'filters' => $request->only(['search']),
        ]);
    }

    /**
     * Show form to create a new meal plan template
     */
    public function create(Request $request)
    {
        $user = $request->user();

        // Get foods available to this coach
        $foods = Food::where(function ($query) use ($user) {
                $query->where('user_id', $user->id)
                      ->orWhereNull('user_id'); // Global foods
            })
            ->where('is_active', true)
            ->orderBy('name')
            ->get(['id', 'name', 'category', 'meal_type', 'calories', 'proteins', 'carbs', 'fats', 'fiber', 'serving_size', 'serving_weight']);

        return Inertia::render('Coach/MealPlans/Templates/Create', [
            'foods' => $foods,
        ]);
    }

    /**
     * Store a new meal plan template
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'calories_target' => 'nullable|integer|min:500|max:10000',
            'protein_target' => 'nullable|integer|min:0|max:500',
            'carbs_target' => 'nullable|integer|min:0|max:1000',
            'fat_target' => 'nullable|integer|min:0|max:500',
            'meals' => 'required|array|min:1',
            'meals.*.name' => 'required|string|max:100',
            'meals.*.time' => 'nullable|string|max:20',
            'meals.*.items' => 'array',
            'meals.*.items.*.food_id' => 'required|exists:foods,id',
            'meals.*.items.*.quantity' => 'required|numeric|min:0.1',
            'meals.*.items.*.notes' => 'nullable|string|max:500',
        ]);

        DB::beginTransaction();

        try {
            $mealPlan = MealPlan::create([
                'user_id' => $request->user()->id,
                'client_id' => null, // Template (no client)
                'name' => $validated['name'],
                'description' => $validated['description'] ?? null,
                'target_calories' => $validated['calories_target'] ?? null,
                'target_proteins' => $validated['protein_target'] ?? null,
                'target_carbs' => $validated['carbs_target'] ?? null,
                'target_fats' => $validated['fat_target'] ?? null,
                'is_active' => true,
            ]);

            // Save meal items
            $this->saveMealItems($mealPlan, $validated['meals']);

            DB::commit();

            return redirect()->route('coach.meal-plans.index')
                ->with('success', 'Plantilla de dieta creada exitosamente.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'Error al guardar: ' . $e->getMessage()]);
        }
    }

    /**
     * Show form to edit a meal plan template
     */
    public function edit(Request $request, MealPlan $mealPlan)
    {
        $user = $request->user();

        // Allow viewing official templates (for cloning) or own templates
        if (!$mealPlan->is_official && ($mealPlan->user_id !== $user->id || $mealPlan->client_id !== null)) {
            abort(403);
        }

        // Load meal items grouped by meal type
        $mealPlan->load(['mealItems.food']);

        $foods = Food::where(function ($query) use ($user) {
                $query->where('user_id', $user->id)
                      ->orWhereNull('user_id');
            })
            ->where('is_active', true)
            ->orderBy('name')
            ->get(['id', 'name', 'category', 'meal_type', 'calories', 'proteins', 'carbs', 'fats', 'fiber', 'serving_size', 'serving_weight']);

        // Transform meal items to the format expected by the form
        $mealsData = $this->transformMealItemsToForm($mealPlan->mealItems);

        return Inertia::render('Coach/MealPlans/Templates/Edit', [
            'template' => array_merge($mealPlan->toArray(), [
                'calories_target' => $mealPlan->target_calories,
                'protein_target' => $mealPlan->target_proteins,
                'carbs_target' => $mealPlan->target_carbs,
                'fat_target' => $mealPlan->target_fats,
                'meals' => $mealsData,
            ]),
            'foods' => $foods,
            'isOfficial' => (bool) $mealPlan->is_official,
        ]);
    }

    /**
     * Update a meal plan template
     */
    public function update(Request $request, MealPlan $mealPlan)
    {
        $user = $request->user();

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'calories_target' => 'nullable|integer|min:500|max:10000',
            'protein_target' => 'nullable|integer|min:0|max:500',
            'carbs_target' => 'nullable|integer|min:0|max:1000',
            'fat_target' => 'nullable|integer|min:0|max:500',
            'meals' => 'required|array|min:1',
            'meals.*.name' => 'required|string|max:100',
            'meals.*.time' => 'nullable|string|max:20',
            'meals.*.items' => 'array',
            'meals.*.items.*.food_id' => 'required|exists:foods,id',
            'meals.*.items.*.quantity' => 'required|numeric|min:0.1',
            'meals.*.items.*.notes' => 'nullable|string|max:500',
        ]);

        // If it's an official template, create a clone for the coach
        if ($mealPlan->is_official) {
            DB::beginTransaction();
            try {
                $clone = MealPlan::create([
                    'user_id' => $user->id,
                    'client_id' => null,
                    'name' => $validated['name'],
                    'description' => $validated['description'] ?? null,
                    'target_calories' => $validated['calories_target'] ?? null,
                    'target_proteins' => $validated['protein_target'] ?? null,
                    'target_carbs' => $validated['carbs_target'] ?? null,
                    'target_fats' => $validated['fat_target'] ?? null,
                    'is_active' => true,
                    'is_official' => false,
                ]);

                $this->saveMealItems($clone, $validated['meals']);
                DB::commit();

                return redirect()->route('coach.meal-plans.index')
                    ->with('success', 'Plantilla oficial duplicada y guardada como tu plantilla personal.');
            } catch (\Exception $e) {
                DB::rollBack();
                return back()->withErrors(['error' => 'Error al duplicar: ' . $e->getMessage()]);
            }
        }

        // Verify ownership for non-official templates
        if ($mealPlan->user_id !== $user->id || $mealPlan->client_id !== null) {
            abort(403);
        }

        DB::beginTransaction();

        try {
            $mealPlan->update([
                'name' => $validated['name'],
                'description' => $validated['description'] ?? null,
                'target_calories' => $validated['calories_target'] ?? null,
                'target_proteins' => $validated['protein_target'] ?? null,
                'target_carbs' => $validated['carbs_target'] ?? null,
                'target_fats' => $validated['fat_target'] ?? null,
            ]);

            // Delete old items and save new ones
            $mealPlan->mealItems()->delete();
            $this->saveMealItems($mealPlan, $validated['meals']);

            DB::commit();

            return redirect()->route('coach.meal-plans.index')
                ->with('success', 'Plantilla de dieta actualizada exitosamente.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'Error al actualizar: ' . $e->getMessage()]);
        }
    }

    /**
     * Delete a meal plan template
     */
    public function destroy(Request $request, MealPlan $mealPlan)
    {
        $user = $request->user();

        // Cannot delete official templates
        if ($mealPlan->is_official) {
            return back()->with('error', 'No se puede eliminar una plantilla oficial.');
        }

        // Verify ownership
        if ($mealPlan->user_id !== $user->id || $mealPlan->client_id !== null) {
            abort(403);
        }

        $mealPlan->delete();

        return redirect()->route('coach.meal-plans.index')
            ->with('success', 'Plantilla de dieta eliminada.');
    }

    /**
     * Save meal items from the form data
     */
    private function saveMealItems(MealPlan $mealPlan, array $meals): void
    {
        $mealTypeMap = [
            'Desayuno' => 'breakfast',
            'Almuerzo' => 'lunch',
            'Cena' => 'dinner',
            'Merienda' => 'snack',
            'Pre-entreno' => 'pre_workout',
            'Post-entreno' => 'post_workout',
            'Media mañana' => 'mid_morning_snack',
            'Media tarde' => 'mid_afternoon_snack',
        ];

        $order = 0;
        foreach ($meals as $meal) {
            $mealType = $mealTypeMap[$meal['name']] ?? 'snack';

            foreach ($meal['items'] ?? [] as $item) {
                if (empty($item['food_id'])) continue;

                $food = Food::find($item['food_id']);
                if (!$food) continue;

                $quantity = $item['quantity'] ?? 1;
                $macros = $food->calculateMacros($quantity * ($food->serving_weight ?: 100));

                MealItem::create([
                    'meal_plan_id' => $mealPlan->id,
                    'food_id' => $item['food_id'],
                    'meal_type' => $mealType,
                    'quantity' => $quantity * ($food->serving_weight ?: 100),
                    'proteins' => $macros['proteins'],
                    'carbs' => $macros['carbs'],
                    'fats' => $macros['fats'],
                    'calories' => $macros['calories'],
                    'order' => $order++,
                ]);
            }
        }
    }

    /**
     * Transform meal items from DB to form format
     */
    private function transformMealItemsToForm($mealItems): array
    {
        $mealTypeLabels = [
            'breakfast' => 'Desayuno',
            'lunch' => 'Almuerzo',
            'dinner' => 'Cena',
            'snack' => 'Merienda',
            'pre_workout' => 'Pre-entreno',
            'post_workout' => 'Post-entreno',
            'mid_morning_snack' => 'Media mañana',
            'mid_afternoon_snack' => 'Media tarde',
            'meal_1' => 'Comida 1',
            'meal_2' => 'Comida 2',
            'meal_3' => 'Comida 3',
        ];

        $grouped = $mealItems->groupBy('meal_type');
        $meals = [];

        foreach ($grouped as $type => $items) {
            $meals[] = [
                'name' => $mealTypeLabels[$type] ?? ucfirst($type),
                'time' => '',
                'items' => $items->map(fn($item) => [
                    'food_id' => $item->food_id,
                    'quantity' => $item->food ? round($item->quantity / ($item->food->serving_weight ?: 100), 1) : 1,
                    'notes' => '',
                ])->toArray(),
            ];
        }

        // If no meals, return default
        if (empty($meals)) {
            $meals = [['name' => 'Desayuno', 'time' => '08:00', 'items' => []]];
        }

        return $meals;
    }
}
