<?php

namespace App\Http\Controllers\Coach;

use App\Http\Controllers\Controller;
use App\Models\ExerciseLibrary;
use App\Models\Workout;
use Illuminate\Http\Request;
use Inertia\Inertia;

/**
 * Controller for managing workout templates (coach-level, not client-specific)
 */
class WorkoutTemplateController extends Controller
{
    /**
     * Display list of workout templates
     */
    public function index(Request $request)
    {
        $user = $request->user();

        $templates = Workout::whereNull('client_id')
            ->where(function ($query) use ($user) {
                $query->where('user_id', $user->id)
                      ->orWhere('is_official', true);
            })
            ->when($request->search, function ($query, $search) {
                $query->where('name', 'like', "%{$search}%");
            })
            ->when($request->type, function ($query, $type) {
                $query->where('type', $type);
            })
            ->orderBy('is_official', 'desc') // Official first
            ->orderBy('created_at', 'desc')
            ->paginate(12)
            ->withQueryString();

        return Inertia::render('Coach/Workouts/Templates/Index', [
            'templates' => $templates,
            'filters' => $request->only(['search', 'type']),
        ]);
    }

    /**
     * Show form to create a new workout template
     */
    public function create(Request $request)
    {
        $coach = $request->user();

        // Get exercises available to this coach
        $exercises = ExerciseLibrary::forCoach($coach->id)
            ->active()
            ->orderBy('name')
            ->get(['id', 'name', 'muscle_group', 'category', 'gif_url', 'coach_id', 'is_system']);

        return Inertia::render('Coach/Workouts/Templates/Create', [
            'exercises' => $exercises,
            'muscleGroups' => ExerciseLibrary::MUSCLE_GROUPS,
            'categories' => ExerciseLibrary::CATEGORIES,
        ]);
    }

    /**
     * Store a new workout template
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'type' => 'nullable|string|max:50',
            'duration_weeks' => 'nullable|integer|min:1|max:52',
            'days' => 'required|array|min:1',
            'days.*.name' => 'required|string|max:100',
            'days.*.exercises' => 'nullable|array',
            'days.*.exercises.*.name' => 'nullable|string|max:255',
            'days.*.exercises.*.exercise_id' => 'nullable|integer',
            'days.*.exercises.*.muscle_group' => 'nullable|string|max:100',
            'days.*.exercises.*.gif_url' => 'nullable|string|max:500',
            'days.*.exercises.*.sets' => 'nullable|integer|min:1',
            'days.*.exercises.*.reps' => 'nullable|string|max:50',
            'days.*.exercises.*.rest' => 'nullable|string|max:50',
            'days.*.exercises.*.rest_seconds' => 'nullable|integer|min:0',
            'days.*.exercises.*.notes' => 'nullable|string|max:500',
            'days.*.exercises.*.video_url' => 'nullable|url|max:500',
            'days.*.exercises.*.block_name' => 'nullable|string|max:100',
            'days.*.exercises.*.order' => 'nullable|integer',
        ]);

        $workout = Workout::create([
            'user_id' => $request->user()->id,
            'client_id' => null, // Template (no client)
            'name' => $validated['name'],
            'description' => $validated['description'] ?? null,
            'type' => $validated['type'] ?? null,
            'duration_weeks' => $validated['duration_weeks'] ?? null,
            'days' => $validated['days'],
            'is_active' => true,
        ]);

        return redirect()->route('coach.workouts.index')
            ->with('success', 'Plantilla de rutina creada exitosamente.');
    }

    /**
     * Show form to edit a workout template
     */
    public function edit(Request $request, Workout $workout)
    {
        $user = $request->user();

        // Allow viewing official templates (for cloning) or own templates
        if (!$workout->is_official && ($workout->user_id !== $user->id || $workout->client_id !== null)) {
            abort(403);
        }

        // Get exercises available to this coach from exercise library
        $exercises = ExerciseLibrary::forCoach($user->id)
            ->active()
            ->orderBy('name')
            ->get(['id', 'name', 'muscle_group', 'category', 'gif_url', 'coach_id', 'is_system']);

        return Inertia::render('Coach/Workouts/Templates/Edit', [
            'template' => $workout,
            'isOfficial' => (bool) $workout->is_official,
            'exercises' => $exercises,
            'muscleGroups' => ExerciseLibrary::MUSCLE_GROUPS,
            'categories' => ExerciseLibrary::CATEGORIES,
        ]);
    }

    /**
     * Update a workout template
     */
    public function update(Request $request, Workout $workout)
    {
        $user = $request->user();

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'duration_weeks' => 'nullable|integer|min:1|max:52',
            'days' => 'required|array|min:1',
            'days.*.name' => 'required|string|max:100',
            'days.*.exercises' => 'array',
            'days.*.exercises.*.name' => 'required|string|max:255',
            'days.*.exercises.*.muscle_group' => 'nullable|string|max:100',
            'days.*.exercises.*.sets' => 'required|integer|min:1',
            'days.*.exercises.*.reps' => 'required|string|max:50',
            'days.*.exercises.*.rest_seconds' => 'nullable|integer|min:0',
            'days.*.exercises.*.notes' => 'nullable|string|max:500',
            'days.*.exercises.*.video_url' => 'nullable|url|max:500',
        ]);

        // If it's an official template, create a clone for the coach instead of updating
        if ($workout->is_official) {
            Workout::create([
                'user_id' => $user->id,
                'client_id' => null,
                'name' => $validated['name'],
                'description' => $validated['description'] ?? null,
                'type' => $workout->type,
                'duration_weeks' => $validated['duration_weeks'] ?? null,
                'days' => $validated['days'],
                'is_active' => true,
                'is_official' => false,
            ]);

            return redirect()->route('coach.workouts.index')
                ->with('success', 'Plantilla oficial duplicada y guardada como tu plantilla personal.');
        }

        // Verify ownership for non-official templates
        if ($workout->user_id !== $user->id || $workout->client_id !== null) {
            abort(403);
        }

        $workout->update([
            'name' => $validated['name'],
            'description' => $validated['description'] ?? null,
            'duration_weeks' => $validated['duration_weeks'] ?? null,
            'days' => $validated['days'],
        ]);

        return redirect()->route('coach.workouts.index')
            ->with('success', 'Plantilla de rutina actualizada exitosamente.');
    }

    /**
     * Delete a workout template
     */
    public function destroy(Request $request, Workout $workout)
    {
        $user = $request->user();

        // Cannot delete official templates
        if ($workout->is_official) {
            return back()->with('error', 'No se puede eliminar una plantilla oficial.');
        }

        // Verify ownership
        if ($workout->user_id !== $user->id || $workout->client_id !== null) {
            abort(403);
        }

        $workout->delete();

        return redirect()->route('coach.workouts.index')
            ->with('success', 'Plantilla de rutina eliminada.');
    }
}
