<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;

class CoachPlan extends Model
{
    protected $fillable = [
        'coach_id',
        'name',
        'slug',
        'seo_title',
        'seo_description',
        'seo_keywords',
        'description',
        'hero_title',
        'hero_subtitle',
        'hero_image',
        'price',
        'currency',
        'currency_symbol',
        'duration_days',
        'features',
        'target_points',
        'faqs',
        'results_images',
        'badge',
        'image',
        'is_active',
        'is_public',
        'max_clients',
        'current_clients',
        // Coach section customization
        'coach_section_title',
        'coach_section_photo',
        'coach_section_bio',
        'coach_section_tagline',
        'coach_section_stats',
        // New landing page fields
        'top_banner_text',
        'show_top_banner',
        'hero_badge',
        'hero_description',
        'hero_cta_text',
        'hero_secondary_cta_text',
        'hero_stats',
        'method_title',
        'method_subtitle',
        'method_items',
        'includes_title',
        'includes_subtitle',
        'includes_items',
        'how_it_works_title',
        'how_it_works_subtitle',
        'how_it_works_steps',
        'nav_links',
        // Pricing section
        'pricing_tiers',
        'pricing_title',
        'pricing_subtitle',
        'pricing_guarantee_text',
        // Results section
        'results_title',
        'results_subtitle',
        // FAQ section
        'faq_title',
        'faq_subtitle',
        // Final CTA
        'final_cta_title',
        'final_cta_subtitle',
        'final_cta_button_text',
        // Draft/Autosave
        'draft_data',
        'last_autosave_at',
        // Landing page colors
        'landing_colors',
    ];

    protected $casts = [
        'price' => 'decimal:2',
        'features' => 'array',
        'target_points' => 'array',
        'faqs' => 'array',
        'results_images' => 'array',
        'is_active' => 'boolean',
        'is_public' => 'boolean',
        'show_top_banner' => 'boolean',
        'hero_stats' => 'array',
        'method_items' => 'array',
        'includes_items' => 'array',
        'how_it_works_steps' => 'array',
        'nav_links' => 'array',
        'pricing_tiers' => 'array',
        'draft_data' => 'array',
        'landing_colors' => 'array',
        'last_autosave_at' => 'datetime',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($plan) {
            if (empty($plan->slug)) {
                $plan->slug = Str::slug($plan->name) . '-' . Str::random(6);
            }
        });
    }

    /**
     * Get the coach who created this plan.
     */
    public function coach(): BelongsTo
    {
        return $this->belongsTo(User::class, 'coach_id');
    }

    /**
     * Get purchases for this plan.
     */
    public function purchases(): HasMany
    {
        return $this->hasMany(CoachPlanPurchase::class);
    }

    /**
     * Get audit logs for this plan.
     */
    public function auditLogs(): HasMany
    {
        return $this->hasMany(CoachPlanAuditLog::class)->orderBy('created_at', 'desc');
    }

    /**
     * Get active purchases.
     */
    public function activePurchases(): HasMany
    {
        return $this->purchases()->where('status', 'paid')->where('ends_at', '>', now());
    }

    /**
     * Check if the plan is available for purchase.
     */
    public function isAvailable(): bool
    {
        if (!$this->is_active) {
            return false;
        }

        if ($this->max_clients !== null && $this->current_clients >= $this->max_clients) {
            return false;
        }

        return true;
    }

    /**
     * Get the payment link for this plan.
     */
    public function getPaymentLink(): ?string
    {
        if (!$this->coach->paypal_enabled) {
            return null;
        }

        if ($this->coach->paypal_me_link) {
            return $this->coach->paypal_me_link . '/' . $this->price;
        }

        return null;
    }

    /**
     * Get the formatted price.
     */
    public function getFormattedPriceAttribute(): string
    {
        return '$' . number_format($this->price, 2);
    }

    /**
     * Get the duration in a human-readable format.
     */
    public function getDurationTextAttribute(): string
    {
        if ($this->duration_days === 7) {
            return '1 semana';
        } elseif ($this->duration_days === 14) {
            return '2 semanas';
        } elseif ($this->duration_days === 30) {
            return '1 mes';
        } elseif ($this->duration_days === 60) {
            return '2 meses';
        } elseif ($this->duration_days === 90) {
            return '3 meses';
        } elseif ($this->duration_days === 180) {
            return '6 meses';
        } elseif ($this->duration_days === 365) {
            return '1 año';
        } else {
            return $this->duration_days . ' días';
        }
    }
}
