<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class DietPlan extends Model
{
    use HasFactory;

    protected $fillable = [
        'meal_plan_id',
        'name',
        'description',
        'days',
        'use_percentages',
        'protein_percent',
        'fat_percent',
        'carb_percent',
        'calorie_adjustment',
        'is_active',
        'order',
    ];

    protected $casts = [
        'days' => 'array',
        'use_percentages' => 'boolean',
        'is_active' => 'boolean',
        'protein_percent' => 'integer',
        'fat_percent' => 'integer',
        'carb_percent' => 'integer',
        'calorie_adjustment' => 'integer',
        'order' => 'integer',
    ];

    /**
     * NOTA: Se eliminaron los appends automáticos para evitar N+1 queries
     * Los totales deben calcularse en el frontend o con eager loading
     */
    // protected $appends = ['total_calories', 'total_protein', 'total_carbs', 'total_fat'];

    public function mealPlan(): BelongsTo
    {
        return $this->belongsTo(MealPlan::class);
    }

    public function mealItems(): HasMany
    {
        return $this->hasMany(MealItem::class)->orderBy('meal_type')->orderBy('order');
    }

    /**
     * Alias for mealItems
     */
    public function items(): HasMany
    {
        return $this->mealItems();
    }

    /**
     * Total calories - usa relación cargada si existe
     */
    public function getTotalCaloriesAttribute(): float
    {
        if ($this->relationLoaded('mealItems')) {
            return $this->mealItems->sum('calories');
        }
        return $this->mealItems()->sum('calories');
    }

    /**
     * Total protein - usa relación cargada si existe
     */
    public function getTotalProteinAttribute(): float
    {
        if ($this->relationLoaded('mealItems')) {
            return $this->mealItems->sum('proteins');
        }
        return $this->mealItems()->sum('proteins');
    }

    /**
     * Total carbs - usa relación cargada si existe
     */
    public function getTotalCarbsAttribute(): float
    {
        if ($this->relationLoaded('mealItems')) {
            return $this->mealItems->sum('carbs');
        }
        return $this->mealItems()->sum('carbs');
    }

    /**
     * Total fat - usa relación cargada si existe
     */
    public function getTotalFatAttribute(): float
    {
        if ($this->relationLoaded('mealItems')) {
            return $this->mealItems->sum('fats');
        }
        return $this->mealItems()->sum('fats');
    }

    /**
     * Calculate total macros from meal items.
     */
    public function calculateTotals(): array
    {
        $items = $this->mealItems;

        return [
            'proteins' => $items->sum('proteins'),
            'carbs' => $items->sum('carbs'),
            'fats' => $items->sum('fats'),
            'calories' => $items->sum('calories'),
        ];
    }
}
