<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class PaymentReceipt extends Model
{
    use HasFactory;

    const STATUS_PENDING = 'pending';
    const STATUS_APPROVED = 'approved';
    const STATUS_REJECTED = 'rejected';

    const TYPE_BANK_TRANSFER = 'bank_transfer';
    const TYPE_WESTERN_UNION = 'western_union';
    const TYPE_PAYPAL = 'paypal';

    protected $fillable = [
        'purchase_id',
        'payment_method_id',
        'type',
        'status',
        'receipt_image',
        'transfer_reference',
        'transfer_date',
        'mtcn_code',
        'sender_name',
        'sent_amount',
        'sent_currency',
        'paypal_transaction_id',
        'paypal_payer_id',
        'paypal_payer_email',
        'reviewed_by',
        'reviewed_at',
        'review_notes',
        'rejection_reason',
    ];

    protected $casts = [
        'transfer_date' => 'datetime',
        'reviewed_at' => 'datetime',
        'sent_amount' => 'decimal:2',
    ];

    /**
     * Get the purchase this receipt belongs to.
     */
    public function purchase(): BelongsTo
    {
        return $this->belongsTo(CoachPlanPurchase::class, 'purchase_id');
    }

    /**
     * Get the payment method used.
     */
    public function paymentMethod(): BelongsTo
    {
        return $this->belongsTo(CoachPaymentMethod::class, 'payment_method_id');
    }

    /**
     * Get the reviewer.
     */
    public function reviewer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'reviewed_by');
    }

    /**
     * Get status label.
     */
    public function getStatusLabelAttribute(): string
    {
        return match ($this->status) {
            self::STATUS_PENDING => 'Pendiente de verificación',
            self::STATUS_APPROVED => 'Aprobado',
            self::STATUS_REJECTED => 'Rechazado',
            default => 'Desconocido',
        };
    }

    /**
     * Get status color for UI.
     */
    public function getStatusColorAttribute(): string
    {
        return match ($this->status) {
            self::STATUS_PENDING => 'yellow',
            self::STATUS_APPROVED => 'green',
            self::STATUS_REJECTED => 'red',
            default => 'gray',
        };
    }

    /**
     * Check if pending.
     */
    public function isPending(): bool
    {
        return $this->status === self::STATUS_PENDING;
    }

    /**
     * Check if approved.
     */
    public function isApproved(): bool
    {
        return $this->status === self::STATUS_APPROVED;
    }

    /**
     * Check if rejected.
     */
    public function isRejected(): bool
    {
        return $this->status === self::STATUS_REJECTED;
    }

    /**
     * Approve the receipt.
     */
    public function approve(int $reviewerId, ?string $notes = null): bool
    {
        $this->update([
            'status' => self::STATUS_APPROVED,
            'reviewed_by' => $reviewerId,
            'reviewed_at' => now(),
            'review_notes' => $notes,
        ]);

        // Update purchase status
        $this->purchase->update([
            'status' => 'paid',
            'paid_at' => now(),
        ]);

        return true;
    }

    /**
     * Reject the receipt.
     */
    public function reject(int $reviewerId, string $reason, ?string $notes = null): bool
    {
        $this->update([
            'status' => self::STATUS_REJECTED,
            'reviewed_by' => $reviewerId,
            'reviewed_at' => now(),
            'rejection_reason' => $reason,
            'review_notes' => $notes,
        ]);

        return true;
    }

    /**
     * Scope for pending receipts.
     */
    public function scopePending($query)
    {
        return $query->where('status', self::STATUS_PENDING);
    }

    /**
     * Scope for approved receipts.
     */
    public function scopeApproved($query)
    {
        return $query->where('status', self::STATUS_APPROVED);
    }
}
