<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class WalletTransaction extends Model
{
    use HasFactory;

    const TYPE_DEPOSIT = 'deposit';
    const TYPE_WITHDRAWAL = 'withdrawal';
    const TYPE_PAYMENT = 'payment';
    const TYPE_REFERRAL_BONUS = 'referral_bonus';
    const TYPE_ADJUSTMENT = 'adjustment';
    const TYPE_REFUND = 'refund';

    const STATUS_PENDING = 'pending';
    const STATUS_COMPLETED = 'completed';
    const STATUS_FAILED = 'failed';
    const STATUS_CANCELLED = 'cancelled';

    protected $fillable = [
        'wallet_id',
        'type',
        'amount',
        'balance_after',
        'description',
        'reference_type',
        'reference_id',
        'payment_method',
        'payment_reference',
        'status',
        'metadata',
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'balance_after' => 'decimal:2',
        'metadata' => 'array',
    ];

    /**
     * Get the wallet that owns the transaction.
     */
    public function wallet(): BelongsTo
    {
        return $this->belongsTo(Wallet::class);
    }

    /**
     * Get the user through wallet.
     */
    public function user()
    {
        return $this->wallet->user();
    }

    /**
     * Check if transaction is a credit (adds money).
     */
    public function isCredit(): bool
    {
        return in_array($this->type, [self::TYPE_DEPOSIT, self::TYPE_REFERRAL_BONUS, self::TYPE_REFUND]);
    }

    /**
     * Check if transaction is a debit (removes money).
     */
    public function isDebit(): bool
    {
        return in_array($this->type, [self::TYPE_WITHDRAWAL, self::TYPE_PAYMENT]);
    }

    /**
     * Get formatted type name.
     */
    public function getTypeNameAttribute(): string
    {
        return match($this->type) {
            self::TYPE_DEPOSIT => 'Depósito',
            self::TYPE_WITHDRAWAL => 'Retiro',
            self::TYPE_PAYMENT => 'Pago',
            self::TYPE_REFERRAL_BONUS => 'Bono Referido',
            self::TYPE_ADJUSTMENT => 'Ajuste',
            self::TYPE_REFUND => 'Reembolso',
            default => ucfirst($this->type),
        };
    }

    /**
     * Get formatted status name.
     */
    public function getStatusNameAttribute(): string
    {
        return match($this->status) {
            self::STATUS_PENDING => 'Pendiente',
            self::STATUS_COMPLETED => 'Completado',
            self::STATUS_FAILED => 'Fallido',
            self::STATUS_CANCELLED => 'Cancelado',
            default => ucfirst($this->status),
        };
    }

    /**
     * Scope for pending transactions.
     */
    public function scopePending($query)
    {
        return $query->where('status', self::STATUS_PENDING);
    }

    /**
     * Scope for completed transactions.
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', self::STATUS_COMPLETED);
    }
}
