<?php

namespace App\Notifications;

use App\Models\Message;
use App\Models\Setting;
use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Str;

class AdminNewSupportMessage extends Notification
{
    // Removed Queueable to process immediately and avoid delays

    /**
     * Create a new notification instance.
     */
    public function __construct(
        public Message $message,
        public string $senderType = 'coach' // 'coach' or 'client'
    ) {}

    /**
     * Get the notification's delivery channels.
     * Database only for instant delivery without queue delays.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $siteName = Setting::get('site_name', 'CoachPro');
        $sender = $this->message->sender;
        $typeLabel = $this->senderType === 'coach' ? 'Coach' : 'Cliente';

        return (new MailMessage)
            ->subject("📩 Nuevo Mensaje de Soporte - {$sender->name}")
            ->greeting("¡Nuevo mensaje de soporte!")
            ->line("Un **{$typeLabel}** ha enviado un mensaje de soporte.")
            ->line('')
            ->line('**Detalles:**')
            ->line("De: {$sender->name} ({$sender->email})")
            ->line("Tipo: {$typeLabel}")
            ->line('')
            ->line('**Mensaje:**')
            ->line('"' . Str::limit($this->message->content, 200) . '"')
            ->action('Ver Mensaje', route('admin.chat.index', ['user' => $sender->id]))
            ->salutation("Sistema de Notificaciones - {$siteName}");
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        $sender = $this->message->sender;
        $typeLabel = $this->senderType === 'coach' ? 'Coach' : 'Cliente';

        return [
            'type' => 'AdminNewSupportMessage',
            'title' => 'Nuevo Mensaje de Soporte',
            'message' => "📩 {$sender->name} ({$typeLabel}) te ha enviado un mensaje de soporte.",
            'sender_type' => $this->senderType,
            'message_id' => $this->message->id,
            'sender_id' => $sender->id,
            'sender_name' => $sender->name,
            'sender_email' => $sender->email,
            'content_preview' => \Illuminate\Support\Str::limit($this->message->content, 100),
            'url' => route('admin.chat.index', ['coach_id' => $sender->id]),
        ];
    }
}
