<?php

namespace App\Notifications;

use App\Models\Order;
use App\Notifications\Traits\ChecksNotificationSettings;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

/**
 * Notification sent to administrators when a store payment fails.
 */
class AdminOrderPaymentFailed extends Notification implements ShouldQueue
{
    use Queueable, ChecksNotificationSettings;

    public function __construct(
        public Order $order,
        public ?string $reason = null,
        public ?string $paymentGateway = null
    ) {}

    public function via(object $notifiable): array
    {
        // Database first to ensure in-app notification is saved even if email fails
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        $message = (new MailMessage)
            ->subject('⚠️ Pago Fallido - Orden #' . $this->order->order_number)
            ->greeting('Alerta de pago fallido')
            ->line('Se ha detectado un pago fallido en la tienda.')
            ->line('')
            ->line('**Detalles del Pedido:**')
            ->line('**Orden:** #' . $this->order->order_number)
            ->line('**Cliente:** ' . $this->order->user->name)
            ->line('**Email:** ' . $this->order->user->email)
            ->line('**Total:** ' . $this->order->formatCurrency($this->order->total))
            ->line('**Método de Pago:** ' . $this->getPaymentMethodLabel());

        if ($this->paymentGateway) {
            $message->line('**Pasarela:** ' . ucfirst($this->paymentGateway));
        }

        if ($this->reason) {
            $message->line('')
                ->line('**Motivo del Error:**')
                ->line($this->reason);
        }

        return $message
            ->line('')
            ->line('El cliente ha sido notificado del problema.')
            ->action('Ver Pedido', route('admin.store.orders.show', $this->order->id))
            ->salutation('Sistema de Notificaciones CoachPro');
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'admin_order_payment_failed',
            'order_id' => $this->order->id,
            'order_number' => $this->order->order_number,
            'customer_name' => $this->order->user->name,
            'customer_email' => $this->order->user->email,
            'total' => $this->order->total,
            'payment_method' => $this->order->payment_method,
            'payment_gateway' => $this->paymentGateway,
            'reason' => $this->reason,
            'message' => "⚠️ Pago fallido en orden #{$this->order->order_number} de {$this->order->user->name}",
            'title' => 'Pago Fallido (Tienda)',
            'url' => route('admin.store.orders.show', $this->order->id),
            'icon' => 'exclamation-circle',
            'priority' => 'high',
        ];
    }

    protected function getPaymentMethodLabel(): string
    {
        $labels = [
            'paypal' => 'PayPal',
            'stripe' => 'Tarjeta (Stripe)',
            'card' => 'Tarjeta de Crédito',
            'transfer' => 'Transferencia Bancaria',
        ];

        return $labels[$this->order->payment_method] ?? $this->order->payment_method;
    }

    public function databaseType(object $notifiable): string
    {
        return 'admin_order_payment_failed';
    }
}
