<?php

namespace App\Notifications;

use App\Models\Setting;
use App\Models\WalletTransaction;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class AdminWalletTransactionPending extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public WalletTransaction $transaction
    ) {}

    public function via(object $notifiable): array
    {
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        $siteName = Setting::get('site_name', 'CoachPro');
        $currency = Setting::get('currency', 'DOP');
        $amount = number_format(abs($this->transaction->amount), 2);
        $type = $this->transaction->type === 'withdrawal' ? 'Retiro' : 'Depósito';
        $userName = $this->transaction->wallet->user->name ?? 'Coach';

        return (new MailMessage)
            ->subject("Nueva Solicitud de {$type} - {$userName}")
            ->greeting("Hola Admin,")
            ->line("Se ha recibido una nueva solicitud de {$type} que requiere aprobación.")
            ->line("**Coach:** {$userName}")
            ->line("**Tipo:** {$type}")
            ->line("**Monto:** \${$amount} {$currency}")
            ->line("**Método:** " . $this->getMethodName())
            ->action('Revisar Transacción', route('admin.wallets.index'))
            ->salutation("Saludos,\nSistema {$siteName}");
    }

    public function toArray(object $notifiable): array
    {
        $amount = number_format(abs($this->transaction->amount), 2);
        $type = $this->transaction->type === 'withdrawal' ? 'retiro' : 'depósito';
        $userName = $this->transaction->wallet->user->name ?? 'Coach';

        return [
            'type' => 'admin_wallet_transaction_pending',
            'transaction_id' => $this->transaction->id,
            'transaction_type' => $this->transaction->type,
            'amount' => abs($this->transaction->amount),
            'user_name' => $userName,
            'message' => "🏦 Nueva solicitud de {$type}: \${$amount} de {$userName}",
            'title' => "Solicitud de {$type} pendiente",
            'url' => route('admin.wallets.index'),
        ];
    }

    protected function getMethodName(): string
    {
        return match($this->transaction->payment_method) {
            'stripe' => 'Tarjeta (Stripe)',
            'paypal' => 'PayPal',
            'bank_transfer' => 'Transferencia Bancaria',
            default => ucfirst($this->transaction->payment_method ?? 'Otro'),
        };
    }
}
