<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class ContactFormSubmission extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * Create a new notification instance.
     */
    public function __construct(
        public array $data
    ) {}

    /**
     * Get the notification's delivery channels.
     */
    public function via(object $notifiable): array
    {
        // Database first to ensure in-app notification is saved even if mail fails
        return ['database', 'mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $typeLabels = [
            'general' => 'Consulta General',
            'support' => 'Soporte Técnico',
            'sales' => 'Ventas',
            'other' => 'Otro',
        ];

        return (new MailMessage)
            ->subject('Nuevo mensaje de contacto: ' . $this->data['subject'])
            ->greeting('¡Nuevo mensaje de contacto!')
            ->line('Has recibido un nuevo mensaje desde el formulario de contacto.')
            ->line('')
            ->line('**Tipo de consulta:** ' . ($typeLabels[$this->data['type']] ?? $this->data['type']))
            ->line('**Nombre:** ' . $this->data['name'])
            ->line('**Email:** ' . $this->data['email'])
            ->line('**Teléfono:** ' . ($this->data['phone'] ?? 'No proporcionado'))
            ->line('')
            ->line('**Asunto:** ' . $this->data['subject'])
            ->line('')
            ->line('**Mensaje:**')
            ->line($this->data['message'])
            ->action('Responder por email', 'mailto:' . $this->data['email'] . '?subject=Re: ' . urlencode($this->data['subject']));
    }

    /**
     * Get the array representation of the notification.
     */
    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'contact_form',
            'title' => 'Nuevo mensaje de contacto',
            'message' => "Nuevo mensaje de {$this->data['name']}: {$this->data['subject']}",
            'data' => [
                'name' => $this->data['name'],
                'email' => $this->data['email'],
                'phone' => $this->data['phone'] ?? null,
                'subject' => $this->data['subject'],
                'type' => $this->data['type'],
            ],
        ];
    }
}
