<?php

namespace App\Notifications;

use App\Models\MealPlan;
use App\Models\Setting;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class MealPlanAssigned extends Notification implements ShouldQueue
{
    use Queueable;

    protected $mealPlan;

    public function __construct(MealPlan $mealPlan)
    {
        $this->mealPlan = $mealPlan;
    }

    public function via($notifiable): array
    {
        return ['database', 'mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $siteName = Setting::get('site_name', 'CoachPro');
        $coachName = $this->mealPlan->client?->user?->name ?? 'Tu Coach';

        return (new MailMessage)
            ->subject("🍽️ Nuevo Plan de Alimentación - {$this->mealPlan->name}")
            ->greeting("¡Hola {$notifiable->name}!")
            ->line("Tu coach {$coachName} te ha asignado un nuevo plan de alimentación.")
            ->line('')
            ->line('**Detalles del Plan:**')
            ->line("Nombre: {$this->mealPlan->name}")
            ->line("Descripción: " . ($this->mealPlan->description ?? 'Sin descripción'))
            ->action('Ver Mi Dieta', route('client.meal-plan.index'))
            ->line('¡Revisa tu nuevo plan alimenticio!')
            ->salutation("Saludos, {$siteName}");
    }

    public function toArray($notifiable): array
    {
        return [
            'type' => 'meal_plan_assigned',
            'meal_plan_id' => $this->mealPlan->id,
            'meal_plan_name' => $this->mealPlan->name,
            'message' => "Tu coach te ha asignado una nueva dieta: {$this->mealPlan->name}",
            'icon' => '🍽️',
            'url' => route('client.meal-plan.index'),
        ];
    }
}
