<?php

namespace App\Services;

use App\Models\Client;
use App\Models\Food;
use App\Models\Invoice;
use App\Models\MealPlan;
use App\Models\PdfSetting;
use App\Models\ProgressPhoto;
use App\Models\User;
use App\Models\Wallet;
use App\Models\WalletTransaction;
use App\Models\Workout;
use App\Models\Setting;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Collection;

class PdfService
{
    /**
     * Get PDF settings (legacy method for backward compatibility).
     */
    private function getSettings(): array
    {
        return [
            'header' => Setting::get('pdf_header', Setting::get('site_name', 'CoachPro')),
            'footer' => Setting::get('pdf_footer', 'www.coachpro.com'),
            'note' => Setting::get('pdf_note', ''),
            'logo' => Setting::get('logo') ? public_path('storage/' . Setting::get('logo')) : null,
        ];
    }

    /**
     * Get enhanced PDF settings for a specific document type.
     */
    private function getPdfSettings(string $documentType): array
    {
        $settings = PdfSetting::getSettingsForType($documentType);
        $globalSettings = $this->getSettings();
        
        // Merge with global settings, preferring document-specific settings
        return array_merge($globalSettings, [
            'pdf_settings' => $settings,
            'primary_color' => $settings['primary_color'] ?? '#3B82F6',
            'secondary_color' => $settings['secondary_color'] ?? '#F97316',
            'accent_color' => $settings['accent_color'] ?? '#10B981',
            'show_logo' => $settings['show_logo'] ?? true,
            'show_date' => $settings['show_date'] ?? true,
            'show_page_numbers' => $settings['show_page_numbers'] ?? true,
            'show_coach_info' => $settings['show_coach_info'] ?? true,
            'show_coach_photo' => $settings['show_coach_photo'] ?? false,
            'show_coach_credentials' => $settings['show_coach_credentials'] ?? true,
            'show_coach_contact' => $settings['show_coach_contact'] ?? true,
            'header_title' => $settings['header_title'] ?? null,
            'header_subtitle' => $settings['header_subtitle'] ?? null,
            'footer_text' => $settings['footer_text'] ?? $globalSettings['footer'] ?? null,
            'footer_note' => $settings['footer_note'] ?? $globalSettings['note'] ?? null,
            'footer_website' => $settings['footer_website'] ?? null,
            'footer_email' => $settings['footer_email'] ?? null,
            'footer_phone' => $settings['footer_phone'] ?? null,
            'custom_css' => $settings['custom_css'] ?? null,
            'watermark_text' => $settings['watermark_text'] ?? null,
            'watermark_enabled' => $settings['watermark_enabled'] ?? false,
            'logo' => $settings['logo'] ? public_path('storage/' . $settings['logo']) : $globalSettings['logo'],
        ]);
    }

    /**
     * Configure PDF options.
     */
    private function configurePdf($pdf, ?array $settings = null): void
    {
        $fontFamily = $settings['pdf_settings']['font_family'] ?? 'DejaVu Sans';
        $paperSize = $settings['pdf_settings']['paper_size'] ?? 'a4';
        
        $pdf->setOptions([
            'isHtml5ParserEnabled' => true,
            'isRemoteEnabled' => true,
            'defaultFont' => $fontFamily,
            'dpi' => 300, // Increased for print quality (WCAG compliant)
            'defaultPaperSize' => $paperSize,
            'isFontSubsettingEnabled' => true,
        ]);
    }

    /**
     * Get coach information for PDFs.
     */
    private function getCoachInfo(?User $coach): array
    {
        if (!$coach) {
            return [];
        }

        // Try multiple photo sources: profile_photo_path, avatar
        $photoPath = null;
        if ($coach->profile_photo_path) {
            $photoPath = public_path('storage/' . $coach->profile_photo_path);
        } elseif ($coach->avatar ?? null) {
            $avatar = $coach->avatar;
            $photoPath = str_starts_with($avatar, 'http') 
                ? $avatar 
                : public_path('storage/' . $avatar);
        }

        return [
            'name' => $coach->name,
            'email' => $coach->email,
            'phone' => $coach->phone ?? null,
            'photo' => $photoPath,
            'credentials' => $coach->credentials ?? $coach->bio ?? null,
            'website' => $coach->website ?? null,
            'specialty' => $coach->specialty ?? null,
        ];
    }


    /**
     * Generate invoice PDF.
     */
    public function generateInvoice(Invoice $invoice): string
    {
        $invoice->load('user', 'subscription.plan');
        $settings = $this->getPdfSettings(PdfSetting::TYPE_INVOICE);

        $pdf = Pdf::loadView('pdf.invoice-new', [
            'invoice' => $invoice,
            'items' => json_decode($invoice->items, true) ?? [],
            'settings' => $settings,
        ]);

        $this->configurePdf($pdf, $settings);
        $orientation = $settings['pdf_settings']['orientation'] ?? 'portrait';
        $pdf->setPaper('a4', $orientation);

        $filename = "invoices/{$invoice->invoice_number}.pdf";
        
        // Use 'public' disk explicitly to store in storage/app/public
        Storage::disk('public')->put($filename, $pdf->output());

        // Update invoice with PDF path
        $invoice->update(['pdf_path' => $filename]);

        return $filename;
    }

    /**
     * Generate workout routine PDF.
     */
    public function generateWorkout(Workout $workout): string
    {
        $workout->load(['client.user', 'exercises']);
        
        // Get coach information
        $coach = $workout->client?->user;

        // Validate required data
        if (!$workout->client) {
            throw new \Exception('Workout must have an associated client');
        }

        if ($workout->exercises->isEmpty()) {
            Log::warning("Generating workout PDF with no exercises for workout ID: {$workout->id}");
        }

        try {
            $settings = $this->getPdfSettings(PdfSetting::TYPE_WORKOUT);
            $coachInfo = $this->getCoachInfo($coach);
            
            $pdf = Pdf::loadView('pdf.workout-new', [
                'workout' => $workout,
                'client' => $workout->client,
                'exercises' => $workout->exercises->sortBy('order'),
                'days' => $this->getDayLabels($workout->days ?? []),
                'settings' => $settings,
                'coach' => $coachInfo,
            ]);

            $this->configurePdf($pdf, $settings);
            $orientation = $settings['pdf_settings']['orientation'] ?? 'portrait';
            $pdf->setPaper('a4', $orientation);

            $directory = 'workouts';
            $filename = "{$directory}/rutina-{$workout->client->id}-{$workout->id}.pdf";
            
            // Use public disk consistently
            $disk = Storage::disk('public');
            
            // Ensure directory exists
            if (!$disk->exists($directory)) {
                $disk->makeDirectory($directory);
            }
            
            // Generate and save PDF
            $disk->put($filename, $pdf->output());
            
            Log::info("Workout PDF generated successfully: {$filename}");

            return $filename;
            
        } catch (\Exception $e) {
            Log::error('Workout PDF Generation Error: ' . $e->getMessage() . ' at ' . $e->getFile() . ':' . $e->getLine());
            throw $e;
        }
    }

    /**
     * Generate meal plan PDF.
     */
    public function generateMealPlan(MealPlan $mealPlan, string $lang = 'es'): string
    {
        $mealPlan->load(['client.user', 'client.portalUser', 'dietPlans.items.food']);
        
        // Get coach information
        $coach = $mealPlan->client?->user;

        // Ensure client exists
        if (!$mealPlan->client) {
            throw new \Exception('Meal plan has no associated client');
        }

        // Get client avatar URL
        $clientAvatarUrl = null;
        $client = $mealPlan->client;
        
        // Check portal user avatar first
        if ($client->portalUser?->avatar) {
            $avatar = $client->portalUser->avatar;
            $clientAvatarUrl = str_starts_with($avatar, 'http') 
                ? $avatar 
                : public_path('storage/' . $avatar);
        } elseif ($client->avatar) {
            $clientAvatarUrl = str_starts_with($client->avatar, 'http') 
                ? $client->avatar 
                : public_path('storage/' . $client->avatar);
        }
        $client->avatar_url = $clientAvatarUrl;

        // Calculate totals directly from loaded data to avoid model accessor issues
        $totals = ['proteins' => 0, 'carbs' => 0, 'fats' => 0, 'calories' => 0];
        
        foreach ($mealPlan->dietPlans as $dietPlan) {
            foreach ($dietPlan->items as $item) {
                $mult = ($item->quantity ?? 100) / 100;
                $totals['calories'] += ($item->calories ?: ($item->food->calories ?? 0)) * $mult;
                $totals['proteins'] += ($item->proteins ?: ($item->food->proteins ?? 0)) * $mult;
                $totals['carbs'] += ($item->carbs ?: ($item->food->carbs ?? 0)) * $mult;
                $totals['fats'] += ($item->fats ?: ($item->food->fats ?? 0)) * $mult;
            }
        }

        try {
            $settings = $this->getPdfSettings(PdfSetting::TYPE_MEAL_PLAN);
            $coachInfo = $this->getCoachInfo($coach);
            
            // Get translations based on language
            $translations = $this->getMealPlanTranslations($lang);
            
            $pdf = Pdf::loadView('pdf.meal-plan-new', [
                'mealPlan' => $mealPlan,
                'client' => $client,
                'dietPlans' => $mealPlan->dietPlans,
                'totals' => $totals,
                'mealLabels' => $this->getMealLabels($lang),
                'settings' => $settings,
                'coach' => $coachInfo,
                'lang' => $lang,
                'translations' => $translations,
            ]);

            $this->configurePdf($pdf, $settings);
            $orientation = $settings['pdf_settings']['orientation'] ?? 'portrait';
            $pdf->setPaper('a4', $orientation);

            $directory = 'meal-plans';
            $filename = "{$directory}/dieta-{$mealPlan->client->id}-{$mealPlan->id}-{$lang}.pdf";
            
            // Use public disk explicitly
            $disk = Storage::disk('public');
            
            // Ensure directory exists
            if (!$disk->exists($directory)) {
                $disk->makeDirectory($directory);
            }
            
            // Generate PDF output and save
            $disk->put($filename, $pdf->output());
            
            Log::info("PDF generated and saved to public disk: {$filename}");

            return $filename;
            
        } catch (\Exception $e) {
            Log::error('PDF Generation Exception: ' . $e->getMessage() . ' at ' . $e->getFile() . ':' . $e->getLine());
            throw $e;
        }
    }
    
    /**
     * Get translations for meal plan PDF.
     */
    private function getMealPlanTranslations(string $lang): array
    {
        $translations = [
            'es' => [
                'meal_plan' => 'Plan de Alimentación',
                'physical_data' => 'Datos Físicos',
                'activity_level' => 'Nivel de Actividad',
                'base_calories' => 'Calorías Base',
                'age' => 'Edad',
                'gender' => 'Sexo',
                'height' => 'Estatura',
                'weight' => 'Peso',
                'male' => 'Masculino',
                'female' => 'Femenino',
                'years' => 'años',
                'activity_factor' => 'Factor de actividad',
                'kcal_day' => 'kcal/día',
                'tdee' => 'TDEE',
                'adjusted_calories' => 'Calorías ajustadas',
                'deficit' => 'Déficit',
                'surplus' => 'Superávit',
                'maintenance' => 'Mantener',
                'proteins' => 'Proteínas',
                'carbs' => 'Carbohidratos',
                'fats' => 'Grasas',
                'calories' => 'Calorías',
                'quantity' => 'Cantidad',
                'food' => 'Alimento',
                'total' => 'Total',
                'notes' => 'Notas',
                'generated_on' => 'Generado el',
                'all_days' => 'Todos los días',
                'days' => 'Días',
                'active_plans' => 'plan(es) activo(s)',
                'serving' => 'Porción',
            ],
            'en' => [
                'meal_plan' => 'Meal Plan',
                'physical_data' => 'Physical Data',
                'activity_level' => 'Activity Level',
                'base_calories' => 'Base Calories',
                'age' => 'Age',
                'gender' => 'Gender',
                'height' => 'Height',
                'weight' => 'Weight',
                'male' => 'Male',
                'female' => 'Female',
                'years' => 'years',
                'activity_factor' => 'Activity factor',
                'kcal_day' => 'kcal/day',
                'tdee' => 'TDEE',
                'adjusted_calories' => 'Adjusted calories',
                'deficit' => 'Deficit',
                'surplus' => 'Surplus',
                'maintenance' => 'Maintenance',
                'proteins' => 'Proteins',
                'carbs' => 'Carbs',
                'fats' => 'Fats',
                'calories' => 'Calories',
                'quantity' => 'Quantity',
                'food' => 'Food',
                'total' => 'Total',
                'notes' => 'Notes',
                'generated_on' => 'Generated on',
                'all_days' => 'All days',
                'days' => 'Days',
                'active_plans' => 'active plan(s)',
                'serving' => 'Serving',
            ],
        ];
        
        return $translations[$lang] ?? $translations['es'];
    }





    /**
     * Generate client statistics/progress PDF.
     */
    public function generateStatistics(Client $client, ?string $period = null): string
    {
        $client->load(['workouts', 'mealPlans', 'progressPhotos']);

        // Note: Measurements feature not yet implemented
        // TODO: Create Measurement model and migration when needed
        $measurements = collect([]);

        // Get progress photos
        $photos = $client->progressPhotos()
            ->orderBy('taken_at', 'desc')
            ->take(8)
            ->get();

        // Calculate stats
        $stats = [
            'total_workouts' => $client->workouts->count(),
            'total_meal_plans' => $client->mealPlans->count(),
            'total_measurements' => $measurements->count(),
            'total_photos' => $client->progressPhotos->count(),
        ];

        $pdf = Pdf::loadView('pdf.statistics', [
            'client' => $client,
            'measurements' => $measurements,
            'photos' => $photos,
            'stats' => $stats,
            'period' => $period ? "Últimos {$period} días" : 'Histórico completo',
            'settings' => $this->getSettings(),
        ]);

        $this->configurePdf($pdf);
        $pdf->setPaper('a4', 'portrait');

        $filename = "statistics/estadisticas-{$client->id}-" . now()->format('Ymd') . ".pdf";
        // Use 'public' disk explicitly to store in storage/app/public
        Storage::disk('public')->put($filename, $pdf->output());

        return $filename;
    }

    /**
     * Generate account statement PDF.
     */
    public function generateAccountStatement(User $user, ?string $startDate = null, ?string $endDate = null): string
    {
        $wallet = $user->wallet;

        // Get transactions
        $transactionsQuery = WalletTransaction::where('wallet_id', $wallet?->id)
            ->orderBy('created_at', 'desc');

        if ($startDate) {
            $transactionsQuery->where('created_at', '>=', $startDate);
        }
        if ($endDate) {
            $transactionsQuery->where('created_at', '<=', $endDate);
        }

        $transactions = $transactionsQuery->get();

        // Get invoices
        $invoicesQuery = Invoice::where('user_id', $user->id)
            ->orderBy('created_at', 'desc');

        if ($startDate) {
            $invoicesQuery->where('created_at', '>=', $startDate);
        }
        if ($endDate) {
            $invoicesQuery->where('created_at', '<=', $endDate);
        }

        $invoices = $invoicesQuery->take(20)->get();

        // Calculate summary
        $summary = [
            'total_in' => $transactions->where('type', 'credit')->sum('amount'),
            'total_out' => $transactions->where('type', 'debit')->sum('amount'),
            'total_transactions' => $transactions->count(),
        ];

        // Period label
        $period = null;
        if ($startDate && $endDate) {
            $period = date('d/m/Y', strtotime($startDate)) . ' - ' . date('d/m/Y', strtotime($endDate));
        } elseif ($startDate) {
            $period = 'Desde ' . date('d/m/Y', strtotime($startDate));
        }

        $pdf = Pdf::loadView('pdf.account-statement', [
            'user' => $user,
            'wallet' => $wallet,
            'transactions' => $transactions,
            'invoices' => $invoices,
            'summary' => $summary,
            'period' => $period,
            'currency' => Setting::get('currency', 'USD'),
            'settings' => $this->getSettings(),
        ]);

        $this->configurePdf($pdf);
        $pdf->setPaper('a4', 'portrait');

        $filename = "statements/cuenta-{$user->id}-" . now()->format('Ymd') . ".pdf";
        // Use 'public' disk explicitly to store in storage/app/public
        Storage::disk('public')->put($filename, $pdf->output());

        return $filename;
    }

    /**
     * Generate foods catalog PDF.
     */
    public function generateFoodsCatalog(?int $coachId = null, bool $groupByCategory = true): string
    {
        $query = Food::query()->orderBy('name');

        if ($coachId) {
            $query->where(function ($q) use ($coachId) {
                $q->where('user_id', $coachId)
                  ->orWhereNull('user_id');
            });
        }

        $foods = $query->get();

        $data = [
            'foods' => $foods,
            'categories' => $foods->pluck('category')->unique()->filter()->count(),
            'avgCalories' => round($foods->avg('calories')),
            'settings' => $this->getSettings(),
        ];

        if ($groupByCategory) {
            $data['groupedFoods'] = $foods->groupBy('category');
        }

        $pdf = Pdf::loadView('pdf.foods', $data);

        $this->configurePdf($pdf);
        $pdf->setPaper('a4', 'portrait');

        $suffix = $coachId ? "-coach-{$coachId}" : '';
        $filename = "foods/catalogo-alimentos{$suffix}-" . now()->format('Ymd') . ".pdf";
        // Use 'public' disk explicitly to store in storage/app/public
        Storage::disk('public')->put($filename, $pdf->output());

        return $filename;
    }

    /**
     * Get day labels in Spanish.
     */
    private function getDayLabels(array $days): array
    {
        $labels = [
            'monday' => 'Lunes',
            'tuesday' => 'Martes',
            'wednesday' => 'Miércoles',
            'thursday' => 'Jueves',
            'friday' => 'Viernes',
            'saturday' => 'Sábado',
            'sunday' => 'Domingo',
        ];

        return array_map(fn($day) => $labels[$day] ?? $day, $days);
    }

    /**
     * Get meal type labels.
     */
    private function getMealLabels(string $lang = 'es'): array
    {
        $labels = [
            'es' => [
                'breakfast' => 'Desayuno',
                'mid_morning_snack' => 'Media Mañana',
                'lunch' => 'Almuerzo',
                'mid_afternoon_snack' => 'Merienda',
                'dinner' => 'Cena',
                'snack' => 'Snack',
                'pre_workout' => 'Pre-entreno',
                'post_workout' => 'Post-entreno',
                'intra_workout' => 'Intra-entreno',
                'desayuno' => 'Desayuno',
                'almuerzo' => 'Almuerzo',
                'merienda' => 'Merienda',
                'cena' => 'Cena',
                'comida_1' => 'Comida 1',
                'comida_2' => 'Comida 2',
                'comida_3' => 'Comida 3',
                'comida_4' => 'Comida 4',
                'comida_5' => 'Comida 5',
                'comida_6' => 'Comida 6',
                'comida_7' => 'Comida 7',
                'comida_8' => 'Comida 8',
            ],
            'en' => [
                'breakfast' => 'Breakfast',
                'mid_morning_snack' => 'Mid-Morning Snack',
                'lunch' => 'Lunch',
                'mid_afternoon_snack' => 'Afternoon Snack',
                'dinner' => 'Dinner',
                'snack' => 'Snack',
                'pre_workout' => 'Pre-Workout',
                'post_workout' => 'Post-Workout',
                'intra_workout' => 'Intra-Workout',
                'desayuno' => 'Breakfast',
                'almuerzo' => 'Lunch',
                'merienda' => 'Snack',
                'cena' => 'Dinner',
                'comida_1' => 'Meal 1',
                'comida_2' => 'Meal 2',
                'comida_3' => 'Meal 3',
                'comida_4' => 'Meal 4',
                'comida_5' => 'Meal 5',
                'comida_6' => 'Meal 6',
                'comida_7' => 'Meal 7',
                'comida_8' => 'Meal 8',
            ],
        ];
        
        return $labels[$lang] ?? $labels['es'];
    }

    /**
     * Download a generated PDF.
     */
    public function download(string $filename)
    {
        // The generate* methods return relative paths (e.g., "workouts/file.pdf")
        // But we store them in "public/workouts/file.pdf"
        $path = "public/{$filename}";
        
        if (!Storage::exists($path)) {
            // Regeneration logic for invoices...
            if (str_starts_with($filename, 'invoices/')) {
                $invoiceNumber = pathinfo($filename, PATHINFO_FILENAME);
                $invoice = Invoice::where('invoice_number', $invoiceNumber)->first();
                
                if ($invoice) {
                    $filename = $this->generateInvoice($invoice);
                    $path = "public/{$filename}";
                }
            }
            
            if (!Storage::exists($path)) {
                // Try checking without 'public/' prefix in case it was stored differently
                if (Storage::exists($filename)) {
                    $path = $filename;
                } else {
                     throw new \Exception("PDF not found: {$filename}");
                }
            }
        }

        return Storage::download($path);
    }

    /**
     * Get PDF stream for inline display.
     */
    public function stream(string $view, array $data = [])
    {
        $data['settings'] = $this->getSettings();
        
        $pdf = Pdf::loadView($view, $data);
        $this->configurePdf($pdf);

        return $pdf->stream();
    }
}
