<?php

namespace App\Services;

use App\Models\PushSubscription;
use App\Models\User;
use Illuminate\Support\Facades\Log;
use Minishlink\WebPush\Subscription;
use Minishlink\WebPush\WebPush;
use Minishlink\WebPush\VAPID;

class PushNotificationService
{
    protected ?WebPush $webPush = null;
    protected string $vapidPublicKey;
    protected string $vapidPrivateKey;
    protected string $vapidSubject;

    public function __construct()
    {
        $this->vapidPublicKey = config('services.push.vapid_public_key', '');
        $this->vapidPrivateKey = config('services.push.vapid_private_key', '');
        $this->vapidSubject = config('services.push.vapid_subject', config('app.url'));

        if ($this->vapidPublicKey && $this->vapidPrivateKey) {
            $this->initWebPush();
        }
    }

    /**
     * Initialize WebPush with VAPID auth.
     */
    protected function initWebPush(): void
    {
        try {
            $this->webPush = new WebPush([
                'VAPID' => [
                    'subject' => $this->vapidSubject,
                    'publicKey' => $this->vapidPublicKey,
                    'privateKey' => $this->vapidPrivateKey,
                ],
            ]);

            // Automatically flush 20 notifications at a time
            $this->webPush->setAutomaticPadding(true);
        } catch (\Exception $e) {
            Log::error('Failed to initialize WebPush', ['error' => $e->getMessage()]);
        }
    }

    /**
     * Check if web push is configured.
     */
    public function isConfigured(): bool
    {
        return $this->webPush !== null;
    }

    /**
     * Send push notification to a user.
     */
    public function sendToUser(User $user, string $title, string $body, array $data = []): int
    {
        if (!$this->isConfigured()) {
            Log::warning('WebPush not configured, skipping notification');
            return 0;
        }

        $subscriptions = PushSubscription::forUser($user->id);
        $sent = 0;

        foreach ($subscriptions as $subscription) {
            if ($this->queueNotification($subscription, $title, $body, $data)) {
                $sent++;
            }
        }

        // Flush all notifications
        $this->flush();

        return $sent;
    }

    /**
     * Queue a push notification for sending.
     */
    protected function queueNotification(PushSubscription $subscription, string $title, string $body, array $data = []): bool
    {
        try {
            $payload = json_encode([
                'title' => $title,
                'body' => $body,
                'icon' => asset('icons/icon-192x192.png'),
                'badge' => asset('icons/icon-72x72.png'),
                'tag' => $data['tag'] ?? 'default',
                'data' => array_merge($data, [
                    'timestamp' => now()->timestamp,
                ]),
            ]);

            $pushSubscription = Subscription::create([
                'endpoint' => $subscription->endpoint,
                'keys' => [
                    'p256dh' => $subscription->p256dh_key,
                    'auth' => $subscription->auth_token,
                ],
            ]);

            $this->webPush->queueNotification($pushSubscription, $payload);

            return true;
        } catch (\Exception $e) {
            Log::error('Failed to queue push notification', [
                'subscription_id' => $subscription->id,
                'error' => $e->getMessage(),
            ]);
            return false;
        }
    }

    /**
     * Flush all queued notifications.
     */
    protected function flush(): void
    {
        if (!$this->webPush) {
            return;
        }

        foreach ($this->webPush->flush() as $report) {
            $endpoint = $report->getRequest()->getUri()->__toString();

            if ($report->isSuccess()) {
                Log::debug('Push notification sent successfully', ['endpoint' => $endpoint]);
                
                // Update last used timestamp
                PushSubscription::where('endpoint', $endpoint)->update(['last_used_at' => now()]);
            } else {
                Log::warning('Push notification failed', [
                    'endpoint' => $endpoint,
                    'reason' => $report->getReason(),
                ]);

                // Deactivate expired subscriptions
                if ($report->isSubscriptionExpired()) {
                    PushSubscription::where('endpoint', $endpoint)->update(['is_active' => false]);
                    Log::info('Push subscription deactivated (expired)', ['endpoint' => $endpoint]);
                }
            }
        }
    }

    /**
     * Send push notification to multiple users.
     */
    public function sendToUsers(array $userIds, string $title, string $body, array $data = []): int
    {
        if (!$this->isConfigured()) {
            return 0;
        }

        $sent = 0;

        foreach ($userIds as $userId) {
            $subscriptions = PushSubscription::where('user_id', $userId)->active()->get();
            
            foreach ($subscriptions as $subscription) {
                if ($this->queueNotification($subscription, $title, $body, $data)) {
                    $sent++;
                }
            }
        }

        $this->flush();

        return $sent;
    }

    /**
     * Send push notification to all users with a specific role.
     */
    public function sendToRole(string $role, string $title, string $body, array $data = []): int
    {
        if (!$this->isConfigured()) {
            return 0;
        }

        $userIds = User::role($role)->pluck('id')->toArray();
        return $this->sendToUsers($userIds, $title, $body, $data);
    }

    /**
     * Generate new VAPID keys.
     */
    public static function generateVapidKeys(): array
    {
        return VAPID::createVapidKeys();
    }
}
