<?php

namespace Database\Seeders;

use App\Models\EmailTemplate;
use Illuminate\Database\Seeder;

class WelcomeEmailTemplatesSeeder extends Seeder
{
    /**
     * Run the database seeds.
     * 
     * Creates email templates for coach welcome messages per plan type.
     */
    public function run(): void
    {
        $templates = [
            // Welcome Trial
            [
                'slug' => 'welcome-trial',
                'name' => 'Bienvenida - Plan Trial',
                'subject' => '🎉 ¡Bienvenido a {{app_name}}, {{coach_name}}! - Plan de Prueba',
                'body' => $this->getTrialTemplate(),
                'variables' => [
                    'app_name',
                    'coach_name',
                    'coach_email',
                    'plan_name',
                    'plan_duration',
                    'max_clients',
                    'login_url',
                ],
                'is_active' => true,
            ],
            // Welcome Basic
            [
                'slug' => 'welcome-basico',
                'name' => 'Bienvenida - Plan Básico',
                'subject' => '🎉 ¡Bienvenido a {{app_name}}, {{coach_name}}! - Plan Básico',
                'body' => $this->getBasicTemplate(),
                'variables' => [
                    'app_name',
                    'coach_name',
                    'coach_email',
                    'plan_name',
                    'plan_price',
                    'max_clients',
                    'login_url',
                ],
                'is_active' => true,
            ],
            // Welcome Premium
            [
                'slug' => 'welcome-premium',
                'name' => 'Bienvenida - Plan Premium',
                'subject' => '👑 ¡Bienvenido a {{app_name}}, {{coach_name}}! - Plan Premium',
                'body' => $this->getPremiumTemplate(),
                'variables' => [
                    'app_name',
                    'coach_name',
                    'coach_email',
                    'plan_name',
                    'plan_price',
                    'login_url',
                ],
                'is_active' => true,
            ],
            // Admin New Coach
            [
                'slug' => 'admin-new-coach',
                'name' => 'Admin - Nuevo Coach Registrado',
                'subject' => '👤 Nuevo Coach: {{coach_name}} - {{app_name}}',
                'body' => $this->getAdminNewCoachTemplate(),
                'variables' => [
                    'app_name',
                    'coach_name',
                    'coach_email',
                    'coach_id',
                    'plan_name',
                    'registration_date',
                    'admin_url',
                ],
                'is_active' => true,
            ],
            // Admin Subscription Purchased
            [
                'slug' => 'admin-subscription-purchased',
                'name' => 'Admin - Nueva Suscripción',
                'subject' => '💰 Nueva Suscripción: {{coach_name}} - Plan {{plan_name}}',
                'body' => $this->getAdminSubscriptionTemplate(),
                'variables' => [
                    'app_name',
                    'coach_name',
                    'coach_email',
                    'plan_name',
                    'plan_price',
                    'payment_method',
                    'subscription_end_date',
                    'admin_url',
                ],
                'is_active' => true,
            ],
        ];

        foreach ($templates as $template) {
            EmailTemplate::updateOrCreate(
                ['slug' => $template['slug']],
                $template
            );
        }

        $this->command->info('✅ Plantillas de email de bienvenida creadas exitosamente.');
    }

    protected function getTrialTemplate(): string
    {
        return <<<'HTML'
<h2>¡Hola {{coach_name}}!</h2>

<p>Te damos la más cordial bienvenida a <strong>{{app_name}}</strong>, la plataforma profesional para entrenadores personales.</p>

<hr>

<h3>🔐 Tus Credenciales de Acceso</h3>
<p><strong>Email:</strong> {{coach_email}}</p>
<p><em>Tu contraseña es la que configuraste al registrarte.</em></p>

<hr>

<h3>📋 Tu Plan: {{plan_name}}</h3>
<p>🎁 <strong>Plan de Prueba Gratuito</strong> - {{plan_duration}} días</p>

<p><strong>Incluye:</strong></p>
<ul>
    <li>✅ Hasta {{max_clients}} clientes activos</li>
    <li>✅ Creación de rutinas de entrenamiento</li>
    <li>✅ Planes de alimentación básicos</li>
    <li>✅ Chat con tus clientes</li>
    <li>✅ Portal de cliente</li>
</ul>

<p><strong>Limitaciones:</strong></p>
<ul>
    <li>❌ Sin marca personalizada (branding)</li>
    <li>❌ Sin exportación de PDFs</li>
    <li>❌ Sin tienda de productos</li>
</ul>

<p>⏰ <strong>Tu período de prueba termina en {{plan_duration}} días.</strong> Actualiza a un plan de pago para continuar sin interrupciones.</p>

<hr>

<h3>🚀 Primeros Pasos</h3>
<ol>
    <li><strong>Inicia sesión</strong> en tu panel de coach</li>
    <li><strong>Completa tu perfil</strong> profesional</li>
    <li><strong>Agrega tu primer cliente</strong></li>
    <li><strong>Crea rutinas y planes de alimentación</strong></li>
</ol>

<p><a href="{{login_url}}" style="display: inline-block; padding: 12px 24px; background-color: #3B82F6; color: white; text-decoration: none; border-radius: 6px;">Acceder a Mi Panel</a></p>

<hr>

<p>¿Tienes dudas? Visita nuestra sección de soporte o contáctanos directamente.</p>

<p>¡Éxito en tu carrera como coach!<br>
El equipo de {{app_name}}</p>
HTML;
    }

    protected function getBasicTemplate(): string
    {
        return <<<'HTML'
<h2>¡Hola {{coach_name}}!</h2>

<p>Te damos la más cordial bienvenida a <strong>{{app_name}}</strong>, la plataforma profesional para entrenadores personales.</p>

<hr>

<h3>🔐 Tus Credenciales de Acceso</h3>
<p><strong>Email:</strong> {{coach_email}}</p>
<p><em>Tu contraseña es la que configuraste al registrarte.</em></p>

<hr>

<h3>📋 Tu Plan: {{plan_name}}</h3>
<p>💼 <strong>Plan Básico</strong> - ${{plan_price}}/mes</p>

<p><strong>Incluye todo del Trial más:</strong></p>
<ul>
    <li>✅ Hasta {{max_clients}} clientes activos</li>
    <li>✅ Exportación de PDFs con tu marca</li>
    <li>✅ Biblioteca de ejercicios personalizada</li>
    <li>✅ Biblioteca de alimentos personalizada</li>
    <li>✅ Historial completo de clientes</li>
    <li>✅ Soporte por email prioritario</li>
</ul>

<p><strong>Limitaciones:</strong></p>
<ul>
    <li>❌ Sin tienda de productos</li>
    <li>❌ Sin planes de coach para venta</li>
</ul>

<p>💡 <strong>Consejo:</strong> El Plan Básico es perfecto para coaches que están comenzando su carrera profesional.</p>

<hr>

<h3>🚀 Primeros Pasos</h3>
<ol>
    <li><strong>Inicia sesión</strong> en tu panel de coach</li>
    <li><strong>Completa tu perfil</strong> profesional</li>
    <li><strong>Agrega tu primer cliente</strong></li>
    <li><strong>Crea rutinas y planes de alimentación</strong></li>
</ol>

<p><a href="{{login_url}}" style="display: inline-block; padding: 12px 24px; background-color: #3B82F6; color: white; text-decoration: none; border-radius: 6px;">Acceder a Mi Panel</a></p>

<hr>

<p>¿Tienes dudas? Visita nuestra sección de soporte o contáctanos directamente.</p>

<p>¡Éxito en tu carrera como coach!<br>
El equipo de {{app_name}}</p>
HTML;
    }

    protected function getPremiumTemplate(): string
    {
        return <<<'HTML'
<h2>¡Hola {{coach_name}}!</h2>

<p>Te damos la más cordial bienvenida a <strong>{{app_name}}</strong>, la plataforma profesional para entrenadores personales.</p>

<hr>

<h3>🔐 Tus Credenciales de Acceso</h3>
<p><strong>Email:</strong> {{coach_email}}</p>
<p><em>Tu contraseña es la que configuraste al registrarte.</em></p>

<hr>

<h3>📋 Tu Plan: {{plan_name}}</h3>
<p>👑 <strong>Plan Premium</strong> - ${{plan_price}}/mes</p>

<p><strong>¡Incluye TODAS las funcionalidades!</strong></p>
<ul>
    <li>✅ Clientes ilimitados</li>
    <li>✅ Tienda de productos integrada</li>
    <li>✅ Vende tus propios planes de entrenamiento</li>
    <li>✅ Sistema de cupones y descuentos</li>
    <li>✅ Programa de referidos con comisiones</li>
    <li>✅ Personalización completa de marca</li>
    <li>✅ Reportes y analytics avanzados</li>
    <li>✅ Soporte prioritario 24/7</li>
    <li>✅ Wallet para gestionar ganancias</li>
</ul>

<p>🌟 <strong>¡Felicidades!</strong> Tienes acceso completo a todas las funcionalidades de la plataforma.</p>

<hr>

<h3>🚀 Primeros Pasos</h3>
<ol>
    <li><strong>Inicia sesión</strong> en tu panel de coach</li>
    <li><strong>Completa tu perfil</strong> profesional</li>
    <li><strong>Configura tus métodos de pago</strong></li>
    <li><strong>Crea tu primer plan para vender</strong></li>
    <li><strong>Agrega productos a tu tienda</strong></li>
</ol>

<p><a href="{{login_url}}" style="display: inline-block; padding: 12px 24px; background-color: #8B5CF6; color: white; text-decoration: none; border-radius: 6px;">Acceder a Mi Panel Premium</a></p>

<hr>

<p>¿Tienes dudas? Como usuario Premium tienes acceso a soporte prioritario 24/7.</p>

<p>¡Éxito en tu carrera como coach!<br>
El equipo de {{app_name}}</p>
HTML;
    }

    protected function getAdminNewCoachTemplate(): string
    {
        return <<<'HTML'
<h2>¡Nuevo Coach Registrado!</h2>

<p>Un nuevo coach se ha registrado en <strong>{{app_name}}</strong>.</p>

<hr>

<h3>👤 Información del Coach</h3>
<ul>
    <li><strong>Nombre:</strong> {{coach_name}}</li>
    <li><strong>Email:</strong> {{coach_email}}</li>
    <li><strong>ID:</strong> #{{coach_id}}</li>
    <li><strong>Plan:</strong> {{plan_name}}</li>
    <li><strong>Fecha de registro:</strong> {{registration_date}}</li>
</ul>

<hr>

<p><a href="{{admin_url}}" style="display: inline-block; padding: 12px 24px; background-color: #10B981; color: white; text-decoration: none; border-radius: 6px;">Ver Detalles del Coach</a></p>

<p>Sistema de Notificaciones de {{app_name}}</p>
HTML;
    }

    protected function getAdminSubscriptionTemplate(): string
    {
        return <<<'HTML'
<h2>¡Nueva Suscripción Recibida!</h2>

<p>Un coach ha adquirido un plan en <strong>{{app_name}}</strong>.</p>

<hr>

<h3>👤 Información del Coach</h3>
<ul>
    <li><strong>Nombre:</strong> {{coach_name}}</li>
    <li><strong>Email:</strong> {{coach_email}}</li>
</ul>

<h3>📋 Detalles de la Suscripción</h3>
<ul>
    <li><strong>Plan:</strong> {{plan_name}}</li>
    <li><strong>Precio:</strong> ${{plan_price}}</li>
    <li><strong>Método de pago:</strong> {{payment_method}}</li>
    <li><strong>Válida hasta:</strong> {{subscription_end_date}}</li>
</ul>

<hr>

<p><a href="{{admin_url}}" style="display: inline-block; padding: 12px 24px; background-color: #3B82F6; color: white; text-decoration: none; border-radius: 6px;">Ver Detalles</a></p>

<p>Sistema de Notificaciones de {{app_name}}</p>
HTML;
    }
}
