/**
 * Currency formatting utilities.
 *
 * Centralized currency formatting functions to avoid duplication across components.
 *
 * @example
 * import { formatPrice, formatCurrency } from '@/Utils/currency';
 *
 * const price = formatPrice(99.99); // "$99.99"
 */

/**
 * Format a value as currency (DOP - Peso Dominicano).
 *
 * @param {number} value - The value to format
 * @param {string} currency - Currency code (default: 'DOP')
 * @param {string} locale - Locale for formatting (default: 'es-DO')
 * @returns {string} Formatted currency string
 */
export const formatPrice = (value, currency = 'DOP', locale = 'es-DO') => {
    if (value === null || value === undefined || isNaN(value)) return 'RD$0.00';

    return 'RD$' + new Intl.NumberFormat('es-DO', {
        minimumFractionDigits: 2,
        maximumFractionDigits: 2,
    }).format(value);
};

/**
 * Format currency in DOP (Peso Dominicano).
 *
 * @param {number} value - The value in DOP
 * @returns {string} Formatted currency string
 */
export const formatCurrency = (value) => {
    if (value === null || value === undefined) return 'RD$0.00';

    return 'RD$' + new Intl.NumberFormat('es-DO', {
        minimumFractionDigits: 2,
        maximumFractionDigits: 2,
    }).format(value);
};

/**
 * Format currency in compact form (DOP).
 *
 * @param {number} value - The value to format
 * @returns {string} Formatted currency string
 */
export const formatCurrencyCompact = (value) => {
    return formatPrice(value);
};

/**
 * Format price with no decimals for display.
 *
 * @param {number} value - The value to format
 * @param {string} currency - Currency code (default: 'DOP')
 * @returns {string} Formatted currency string without decimals
 */
export const formatPriceRounded = (value, currency = 'DOP') => {
    if (value === null || value === undefined || isNaN(value)) return 'RD$0';

    return 'RD$' + new Intl.NumberFormat('es-DO', {
        minimumFractionDigits: 0,
        maximumFractionDigits: 0,
    }).format(value);
};

/**
 * Format a number with thousand separators.
 *
 * @param {number} value - The value to format
 * @param {number} decimals - Number of decimal places (default: 0)
 * @returns {string} Formatted number string
 */
export const formatNumber = (value, decimals = 0) => {
    if (value === null || value === undefined || isNaN(value)) return '0';

    return new Intl.NumberFormat('en-US', {
        minimumFractionDigits: decimals,
        maximumFractionDigits: decimals,
    }).format(value);
};

/**
 * Parse a currency string to number.
 *
 * @param {string} value - The currency string
 * @returns {number} Parsed number value
 */
export const parseCurrency = (value) => {
    if (!value) return 0;
    return parseFloat(String(value).replace(/[^0-9.-]+/g, '')) || 0;
};

export default {
    formatPrice,
    formatCurrency,
    formatCurrencyCompact,
    formatPriceRounded,
    formatNumber,
    parseCurrency,
};
