/**
 * Date and time formatting utilities.
 * 
 * Centralized date formatting functions to avoid duplication across components.
 * 
 * @example
 * import { formatDate, formatDateTime, formatRelativeTime } from '@/Utils/formatters';
 * 
 * const formatted = formatDate('2024-01-15'); // "15 ene 2024"
 */

/**
 * Format a date string to localized date format.
 * 
 * @param {string|Date} date - The date to format
 * @param {Object} options - Intl.DateTimeFormat options
 * @returns {string} Formatted date string
 */
export function formatDate(date, options = {}) {
    if (!date) return '';
    
    const defaultOptions = {
        day: '2-digit',
        month: 'short',
        year: 'numeric',
    };
    
    try {
        return new Date(date).toLocaleDateString('es-ES', { ...defaultOptions, ...options });
    } catch (e) {
        return '';
    }
}

/**
 * Format a date string to include time.
 * 
 * @param {string|Date} date - The date to format
 * @returns {string} Formatted date and time string
 */
export function formatDateTime(date) {
    if (!date) return '';
    
    try {
        return new Date(date).toLocaleDateString('es-ES', {
            day: '2-digit',
            month: 'short',
            year: 'numeric',
            hour: '2-digit',
            minute: '2-digit',
        });
    } catch (e) {
        return '';
    }
}

/**
 * Format a date to relative time (e.g., "hace 2 días").
 * 
 * @param {string|Date} date - The date to format
 * @returns {string} Relative time string
 */
export function formatRelativeTime(date) {
    if (!date) return '';
    
    try {
        const rtf = new Intl.RelativeTimeFormat('es', { numeric: 'auto' });
        const now = new Date();
        const then = new Date(date);
        const diffInSeconds = Math.floor((then - now) / 1000);
        
        const intervals = [
            { unit: 'year', seconds: 31536000 },
            { unit: 'month', seconds: 2592000 },
            { unit: 'week', seconds: 604800 },
            { unit: 'day', seconds: 86400 },
            { unit: 'hour', seconds: 3600 },
            { unit: 'minute', seconds: 60 },
            { unit: 'second', seconds: 1 },
        ];
        
        for (const { unit, seconds } of intervals) {
            const interval = Math.floor(diffInSeconds / seconds);
            if (Math.abs(interval) >= 1) {
                return rtf.format(interval, unit);
            }
        }
        
        return rtf.format(0, 'second');
    } catch (e) {
        return '';
    }
}

/**
 * Get the current year.
 * 
 * @returns {number} Current year
 */
export function getCurrentYear() {
    return new Date().getFullYear();
}

/**
 * Calculate days remaining until a date.
 * 
 * @param {string|Date} endDate - The end date
 * @returns {number} Number of days remaining (negative if past)
 */
export function getDaysRemaining(endDate) {
    if (!endDate) return 0;
    
    const end = new Date(endDate);
    const now = new Date();
    const diffTime = end - now;
    return Math.ceil(diffTime / (1000 * 60 * 60 * 24));
}

/**
 * Check if a date is in the past.
 * 
 * @param {string|Date} date - The date to check
 * @returns {boolean} True if the date is in the past
 */
export function isPastDate(date) {
    if (!date) return false;
    return new Date(date) < new Date();
}

/**
 * Check if a date is today.
 * 
 * @param {string|Date} date - The date to check
 * @returns {boolean} True if the date is today
 */
export function isToday(date) {
    if (!date) return false;
    const d = new Date(date);
    const today = new Date();
    return d.toDateString() === today.toDateString();
}

/**
 * Format a duration in days to a human-readable string.
 * 
 * @param {number} days - Number of days
 * @returns {string} Formatted duration string
 */
export function formatDuration(days) {
    if (!days || days <= 0) return '';
    
    if (days === 1) return '1 día';
    if (days < 7) return `${days} días`;
    if (days === 7) return '1 semana';
    if (days < 30) return `${Math.floor(days / 7)} semanas`;
    if (days === 30) return '1 mes';
    if (days < 365) return `${Math.floor(days / 30)} meses`;
    if (days === 365) return '1 año';
    return `${Math.floor(days / 365)} años`;
}

export default {
    formatDate,
    formatDateTime,
    formatRelativeTime,
    getCurrentYear,
    getDaysRemaining,
    isPastDate,
    isToday,
    formatDuration,
};
